#ifndef __ATO_UTILCORE_PRIVATE_H__
#define __ATO_UTILCORE_PRIVATE_H__

#include <stddef.h>

#include "atobase/utilcore.h"

/* ! @defgroup atobase_api_putil PUtil
@ingroup atobase_api_private

@{
*/

/**
The ATO_ASSERT_ISNOTALLOCATED() function macro is used whenever an API method in an SDK has to allocate memory.
It checks that the memory (*obj) is initialised to NULL. This catches some circumstances where
allocated memory may be leaked.
This is only enabled if the macro ATO_CHECK_ISNOTALLOCATED is defined.
Otherwise just check the pointer itself is not NULL (a NULL argument check within the API calls).
The preferred mechanism is to use a memory profiler such as Valgrind or Purify - however
it may be useful during debugging and testing.
*/
#define ATO_CHECK_ISNOTALLOCATED
#ifdef ATO_CHECK_ISNOTALLOCATED
#define ATO_ASSERT_ISNOTALLOCATED(obj) (assert((obj) != NULL && *(obj) == NULL))
#else
#define ATO_ASSERT_ISNOTALLOCATED(obj) (assert((obj) != NULL)
#endif

/** @internal
Strip the const attribute from types. For example "const char *" becomes "char *'.
This is useful for passing const types to third party libraries that don't support them, yet are
treating them as const. This, obviously, should be used with care and ONLY as a last resort if you are
sure the method is treating it as a const.
*/
#define ATO_REMOVE_CONST(x, type) ((type)(uintptr_t)(const void *)(x))
//#define ATO_REMOVE_CONST(x, type) ((type)(size_t)(x))

#ifdef __cplusplus
extern "C" {
#endif

/** @internal
Free v if not NULL and return NULL but 0 out the string first.
*/
ATO_EXPORT char *ato_freeclr(char *v);

/** @internal
Checks if a file exists.
@param filepath the relative or absolute path to the file.
@return TRUE or FALSE.
*/
ATO_EXPORT bool ato_fileexists(const char *filepath);

/**
If the string is NULL, just return 0.
@param src the string to check
@return Return 0 if the string is NULL, otherwise return the strlen()
*/
ATO_EXPORT size_t ato_strlen(const char *src);

/**
Allocate a string of sufficient size to hold the catanation of the 3 non-NULL strings - caller must free.
If any string is NULL, just ignore it.
@param src1 first string
@param src2 second string
@param src3 third string
@return a new string
*/
ATO_EXPORT char *ato_strcat(const char *src1, const char *src2, const char *src3);

/**
Allocate a string of sufficient size to hold the catanation of the 4 non-NULL strings - caller must free.
If any string is NULL, just ignore it.
@param src1 first string
@param src2 second string
@param src3 third string
@param src4 fourth string
@return a new string
*/
ATO_EXPORT char *ato_strcat4(const char *src1, const char *src2, const char *src3, const char *src4);

/**
Allocate a string of sufficient size to hold the duplicate of src plus a padding length - caller must free.
Will be null terminated.
If the string is NULL, just return NULL.
@param src the string to duplicate
@padlen padding length (can be 0)
@return a new string
*/
ATO_EXPORT char *ato_strdup(const char *src, size_t padlen);
/**
Allocate a string of sufficient size to hold the duplicate of n chars of src plus a padding length - caller must free.
Will be null terminated.
If the string is NULL, just return NULL.
@param src the string to duplicate
@padlen padding length (can be 0)
@len length to copy
@return a new string
*/
ATO_EXPORT char *ato_strndup(const char *src, size_t padlen, size_t len);

/**
Allocate memory size 'len' to hold the duplicate of 'src' plus a padding length - caller must free.
If the string is NULL, just return NULL.
@param src the memory block to duplicate
@param len the size of the memory block
@param padlen padding length (can be 0)
@return a new memory block
*/
ATO_EXPORT void *ato_memdup(const void *src, size_t len, size_t padlen);

/**
Thread safe way to convert a set of "strings" seperated by delim to an array of "strings" terminated by a NULL entry.
Use ato_strtokfree() to free.

@param dst the address of a char** to allocate
@param src source string such as "a,b,c"
@param delim the delimeter char such as ','
@return size of the array (alhough the last entry is NULL)
*/
ATO_EXPORT size_t ato_strtok2array(char ***dst, const char *src, const char *delim);

/**
Free the array allocation by ato_strtok2array().
@param dst the array to de-allocate
*/
ATO_EXPORT void ato_strtokfree(char *dst[]);

ATO_EXPORT char *ato_strtok(char *str, const char *delimiters, char **context);

/**

*/
ATO_EXPORT char *ato_pathcat(const char *path1, const char *path2);
ATO_EXPORT time_t ato_str2time(const char *src);
ATO_EXPORT void ato_gmtime(time_t *timep, struct tm *result);
ATO_EXPORT void ato_localtime(time_t *timep, struct tm *result);
ATO_EXPORT size_t ato_strftime_utc(char *buf, size_t sizeinbytes, time_t *timep, bool isUtc); // format="%Y/%m/%d-%H:%M:%S"
ATO_EXPORT char *ato_host(char **buffer, const char *url);
ATO_EXPORT bool ato_isnullorempty(const char *str);
ATO_EXPORT bool ato_isnulloremptyws(const char *str, size_t len);
ATO_EXPORT size_t ato_trim(char *str, size_t len);
ATO_EXPORT char *ato_substitute(char **dst, const char *src, const char *value);
ATO_EXPORT char *ato_substitutevar(char **dst, const char *src, const char *name, const char *value);
ATO_EXPORT bool ato_streq(const char *src, const char *src2);
ATO_EXPORT bool ato_streqi(const char *src, const char *src2);
ATO_EXPORT bool ato_strstartswith(const char *src, const char *src2);
ATO_EXPORT bool ato_strstartswithi(const char *src, const char *src2);
ATO_EXPORT char *ato_itoa(char *dst, int value);
ATO_EXPORT const char *ato_value(const char *value, const char *defvalue);
ATO_EXPORT const char *ato_valuec(const char *value, const char *defvalue);
ATO_EXPORT int ato_value_int(const char *value, int defvalue);
ATO_EXPORT bool ato_value_bool(const char *value, bool defvalue);
ATO_EXPORT void *ato_value_v(void *value, void *defvalue);
ATO_EXPORT char *ato_striff(char *str1, char *str2);

// caller must free returned value if not NULL
ATO_EXPORT char *ato_substr(const char *str, const char *start, const char *end);
ATO_EXPORT char *ato_substrmatch(const char *str, const char *matchstart, const char *matchend);

ATO_EXPORT char *ato_stripouterxml(char *text);

#ifdef __cplusplus
}
#endif

/* ! @} */

#endif /*__ATO_UTILCORE_PRIVATE_H__*/
