/*
This module is guaranteed not to generate exceptions.
Methods will either succeed, assert, or return an errcode (if returning int).
*/
#include <stdlib.h>
#include <assert.h>
#include <string.h>
#include <stdio.h>
#include <time.h>
#include <stdarg.h>

#include "atointernal.h"
#include "atotypes.h"
#include "atolst.h"

#include "atostr.h"

#include "atoutil.h"

static const char *_library = ATO_BASE_LIBRARY;
static const char *_module = ATO_BASE_MODULE_STR;
static unsigned long _moduleid = ATO_BASE_MODULEID_STR;

static int _loglevel = ATO_LOG_WARN;

struct _ato_String {
    char *value;
    const char *valueconst;
    size_t len;
    ato_str_freefunction *callback;
    void *callback_data;
    ato_eStrFormat format;
    bool iscstr;
    ato_List *lst;
    size_t lst_len;
};

/*********************************************************************************/
static void _setloglevel(ato_eLoglevel level)
{
    _loglevel = level;
}

/*********************************************************************************/
int ato__str_init(void)
{
    static bool invoked = FALSE;
    if (invoked) return ATO_ERR_OK;
    invoked = TRUE;

    ato_initfnloglevel(_library, _module, _moduleid, _loglevel, _setloglevel);
    return ATO_ERR_OK;
}
void ato__str_deinit(void)
{
}

/*********************************************************************************/
static void _free_str_contents(ato_String *str)
{
    if (str->callback && str->callback_data) {
        str->callback(str->callback_data);
    } else if (str->value != NULL) {
        free((char *)str->value);
    }
    str->valueconst = NULL;
    str->value = NULL;
    str->len = 0;
}

/*********************************************************************************/
ato_String *ato_str_create(ato_String **obj, char *value, size_t len, bool iscstr)
{
    ATO_ASSERT_ISNOTALLOCATED(obj);
    *obj = calloc(1, sizeof(ato_String));
    assert(*obj != NULL);
    ato_str_set(*obj, value, len, iscstr);
    return *obj;
}
ato_String *ato_str_createconst(ato_String **obj, const char *value, size_t len, bool iscstr)
{
    ATO_ASSERT_ISNOTALLOCATED(obj);
    *obj = calloc(1, sizeof(ato_String));
    assert(*obj != NULL);
    ato_str_setconst(*obj, value, len, iscstr);
    return *obj;
}

void ato_str_setformat(ato_String *str, ato_eStrFormat format)
{
    if (str == NULL) return;
    str->format = format;
}

void ato_str_setfreefn(ato_String *str, ato_str_freefunction callback, void *data)
{
    str->callback = callback;
    str->callback_data = data;
}

void *ato_str_free(ato_String *str)
{
    if (str == NULL) { return NULL; }
    _free_str_contents(str);
    str->lst = ato_lst_free(str->lst);
    free(str);
    return NULL;
}

void ato_str_init(ato_String *str, int value) {
    if (str == NULL) { return; }
    if (str->value == NULL) { return; }
    memset(str->value, value, str->len);
}

void ato_str_dup2(ato_String **str, const ato_String *src, bool iscstr)
{
    size_t len = 0;
    ATO_ASSERT_ISNOTALLOCATED(str);
    if (src == NULL) return;
    *str = calloc(1, sizeof(ato_String));
    assert(*str != NULL);
    (*str)->iscstr = src->iscstr || iscstr;
    (*str)->len = len = src->len;
    if (src->iscstr) {
        len++; // get the '\0'
    }
    (*str)->valueconst = (*str)->value = ato_memdup(src->valueconst, len, 0);
    ((*str)->value)[len] = '\0';
}

void ato_str_dup(ato_String **str, const ato_String *src)
{
    ato_str_dup2(str, src, FALSE);
}

static void _str_set(ato_String *str, const char *valueconst, size_t len, bool iscstr)
{
    _free_str_contents(str);
    str->len = len;
    str->valueconst = valueconst;
    str->value = NULL;
    str->iscstr = iscstr && (valueconst != NULL);
    if (str->iscstr) {
        assert(valueconst[len] == '\0');
    }
}

void ato_str_set(ato_String *str, char *value, size_t len, bool iscstr)
{
    _str_set(str, value, len, iscstr);
    str->value = value;
}

void ato_str_setconst(ato_String *str, const char *value, size_t len, bool iscstr)
{
    _str_set(str, value, len, iscstr);
}

const char *ato_str_value(const ato_String *str)
{
    if (str == NULL) return NULL;
    return str->valueconst;
}

size_t ato_str_len(const ato_String *str)
{
    if (str == NULL) return 0;
    return str->len;
}

bool ato_str_isnullorempty(ato_String *str)
{
    if (str == NULL) return TRUE;
    return str->len == 0 || ato_isnullorempty(str->valueconst);
}

bool ato_str_iscstr(ato_String *str)
{
    if (str == NULL || str->len == 0) return FALSE;
    return str->iscstr;
}

ato_eStrFormat ato_str_type(ato_String *str)
{
    if (str == NULL) return ATO_STR_FORMAT_UNDEFINED;
    return str->format;
}

char *ato_str_valueasowner(ato_String *str)
{
    if (str == NULL) return NULL;
    else {
        char *v = str->value;
        str->value = NULL;
        return v;
    }
}

bool ato_str_eq(ato_String *str, ato_String *str2)
{
    if (str == NULL && str2 == NULL)
        return TRUE;
    if ((str == NULL && str2 != NULL) || (str != NULL && str2 == NULL))
        return FALSE;
    if (str->valueconst == str2->valueconst)
        return TRUE;
    if (str->len != str2->len)
        return FALSE;
    return memcmp(str->valueconst, str2->valueconst, str2->len) == 0;
}

bool ato_str_eqv(ato_String *str, const char *cstr)
{
    size_t len = 0;
    if (str == NULL && cstr == NULL)
        return TRUE;
    if ((str == NULL && cstr != NULL) || (str != NULL && cstr == NULL))
        return FALSE;
    if (str->valueconst == cstr)
        return TRUE;
    len = strlen(cstr);
    if (str->len != len)
        return FALSE;
    return memcmp(str->valueconst, cstr, len) == 0;
}

char *ato_str_cstr(ato_String *str) {
    char *dst = NULL;
    int len = ato_str_len(str);
    if (str == NULL) return NULL;
    dst = calloc(len + 1, sizeof(char));
    memcpy(dst, ato_str_value(str), len);
    return dst;
}

void ato_str_add(ato_String *str, const char *cstr) {
    if (cstr == NULL || *cstr == '\0') { return; }
    if (str->lst == NULL) {
        ato_lst_create(&(str->lst));
        ato_lst_addconst(str->lst, NULL, str->valueconst);
        str->lst_len = str->len;
    }
    ato_lst_addconst(str->lst, NULL, cstr);
    str->lst_len += strlen(cstr);
}

void ato_str_add_complete(ato_String *str) {
    if (str == NULL) { return; }
    if (str->lst != NULL && str->lst_len > 0) {
        ato_ListNode *node = NULL;
        char *newcstr = calloc(str->lst_len + 1, sizeof(char));
        while ((node = ato_lst_next(str->lst, node)) != NULL) {
            const char *cstr = ato_lst_valueorconst(node);
            if (cstr != NULL) { strcat(newcstr, cstr); }
        }
        ato_str_set(str, newcstr, str->lst_len, TRUE);
    }
    str->lst = ato_lst_free(str->lst);
    str->lst_len = 0;
}
