#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>
#include <time.h>
#include <stdarg.h>

#include "libxml/tree.h"
#include "libxml/xmlmemory.h"
#include "libxml/parser.h"

#include "xmlsec/xmlsec.h"
#include "xmlsec/xmltree.h"
#include "xmlsec/xmldsig.h"
#include "xmlsec/xmlenc.h" // for decrypt only
#include "xmlsec/crypto.h"
#include "xmlsec/parser.h"

#include "atobase/private/pall.h"
#include "atobnet/private/pall.h"

#include "atointernal.h"
#include "atotypes.h"
#include "atostmtkn.h"
#include "atostm.h"

static const char *_library = ATO_STM_LIBRARY;
static const char *_module = ATO_STM_MODULE_STM;
static unsigned long _moduleid = ATO_STM_MODULEID_STM;

static ato_eLoglevel _loglevel = ATO_LOG_WARN;
static bool _logdumpfile = FALSE;

/*********************************************************************************/
struct _ato_Stm {
    size_t timeout;
    size_t timeoutconnect;
    char *url;
    char *contenttype;
    ato_String *xtemplate;
    ato_List *vars;
    size_t vars_total_length;
    int headermin_adj; // adjustment for testing - normally 0
    int bodymin_adj;
    int headermin_ttl; // time to live - default 5 mins
    int bodymin_ttl;
    char *rp_p8;
};

/*********************************************************************************/
static const char *MessageIDName = "/s:Envelope/s:Header/a:MessageID";
static const char *ToName = "/s:Envelope/s:Header/a:To";
static const char *CreatedHdrName = "/s:Envelope/s:Header/o:Security/u:Timestamp/u:Created";
static const char *ExpiresHdrName = "/s:Envelope/s:Header/o:Security/u:Timestamp/u:Expires";
static const char *CreatedBodyName = "/s:Envelope/s:Body/wst:RequestSecurityToken/wst:Lifetime/wsu:Created";
static const char *ExpiresBodyName = "/s:Envelope/s:Body/wst:RequestSecurityToken/wst:Lifetime/wsu:Expires";
static const char *BinarySecurityTokenName = "/s:Envelope/s:Header/o:Security/o:BinarySecurityToken";
static const char *AddressName = "/s:Envelope/s:Body/wst:RequestSecurityToken/wsp:AppliesTo/wsa:EndpointReference/wsa:Address";

// Applies to times in Body
#define ATO_STM_EXPIRY_MINS_MIN 5
#define ATO_STM_EXPIRY_MINS_MAX 30
#define ATO_STM_EXPIRY_MINS_DEFAULT 5

/*********************************************************************************/
static void _ato_free_vars(ato_Stm *stm)
{
    ato_ListNode *curr = NULL;
    if (stm == NULL) return;
    if (stm->vars == NULL) return;

    while ((curr = ato_lst_next(stm->vars, curr)) != NULL) {
        char *value = ato_lst_value(curr);
        if (value != NULL) { ato_free(value); }
    }
    stm->vars = ato_lst_free(stm->vars);
    stm->vars_total_length = 0;
}

static void _ato_free(ato_Stm *stm)
{
    if (stm == NULL) return;

    if (stm->contenttype != NULL) { free(stm->contenttype); stm->contenttype = NULL; }
    if (stm->url != NULL) { free(stm->url); stm->url = NULL; }
    if (stm->rp_p8 != NULL) { free(stm->rp_p8); stm->rp_p8 = NULL; }

    ato_str_free(stm->xtemplate);
    _ato_free_vars(stm);
    free(stm);
}

static void _setns(ato_Ctx *ctx, ato_Xml *xml)
{
    assert(xml != NULL);
    ato_xml_addnamespace(ctx, xml, "s", "http://www.w3.org/2003/05/soap-envelope");
    ato_xml_addnamespace(ctx, xml, "a", "http://www.w3.org/2005/08/addressing");
    ato_xml_addnamespace(ctx, xml, "u", "http://docs.oasis-open.org/wss/2004/01/oasis-200401-wss-wssecurity-utility-1.0.xsd");
    ato_xml_addnamespace(ctx, xml, "o", "http://docs.oasis-open.org/wss/2004/01/oasis-200401-wss-wssecurity-secext-1.0.xsd");
    ato_xml_addnamespace(ctx, xml, "i", "http://schemas.xmlsoap.org/ws/2005/05/identity");
    ato_xml_addnamespace(ctx, xml, "sig", "http://www.w3.org/2000/09/xmldsig#");
    ato_xml_addnamespace(ctx, xml, "wst", "http://docs.oasis-open.org/ws-sx/ws-trust/200512");
    ato_xml_addnamespace(ctx, xml, "wsu", "http://docs.oasis-open.org/wss/2004/01/oasis-200401-wss-wssecurity-utility-1.0.xsd");
    ato_xml_addnamespace(ctx, xml, "wsa", "http://www.w3.org/2005/08/addressing");
    ato_xml_addnamespace(ctx, xml, "wsp", "http://schemas.xmlsoap.org/ws/2004/09/policy");
}

static void _printstuff(ato_Ctx *ctx, ato_Log *log, ato_Xml *xml)
{
    const char *function = "_printstuff";
    char *MessageID = NULL, *To = NULL, *Created = NULL, *Expires = NULL, *CreatedS = NULL, *ExpiresS = NULL,
        *BinarySecurityToken = NULL, *Address = NULL;

    ato_xml_nodevalue(xml, NULL, MessageIDName, &MessageID, FALSE);
    ATO_LOG_MSGV(log, ATO_LOG_DEBUG, "MessageID=%s", MessageID);
    ato_xml_nodevalue(xml, NULL, ToName, &To, FALSE);
    ATO_LOG_MSGV(log, ATO_LOG_DEBUG, "To=%s", To);
    ato_xml_nodevalue(xml, NULL, CreatedHdrName, &Created, FALSE);
    ATO_LOG_MSGV(log, ATO_LOG_DEBUG, "Created=%s", Created);
    ato_xml_nodevalue(xml, NULL, ExpiresHdrName, &Expires, FALSE);
    ATO_LOG_MSGV(log, ATO_LOG_DEBUG, "Expires=%s", Expires, FALSE);
    ato_xml_nodevalue(xml, NULL, CreatedBodyName, &CreatedS, FALSE);
    ATO_LOG_MSGV(log, ATO_LOG_DEBUG, "CreatedS=%s", CreatedS);
    ato_xml_nodevalue(xml, NULL, ExpiresBodyName, &ExpiresS, FALSE);
    ATO_LOG_MSGV(log, ATO_LOG_DEBUG, "ExpiresS=%s", ExpiresS);
    ato_xml_nodevalue(xml, NULL, AddressName, &Address, FALSE);
    ATO_LOG_MSGV(log, ATO_LOG_DEBUG, "Address=%s", Address);
    ato_xml_nodevalue(xml, NULL, BinarySecurityTokenName, &BinarySecurityToken, FALSE);
    ATO_LOG_MSGV(log, ATO_LOG_DEBUG, "BinarySecurityToken=%s", BinarySecurityToken);

    MessageID = ato_free(MessageID);
    Created = ato_free(Created);
    Expires = ato_free(Expires);
    CreatedS = ato_free(CreatedS);
    ExpiresS = ato_free(ExpiresS);
    BinarySecurityToken = ato_free(BinarySecurityToken);
    Address = ato_free(Address);
    To = ato_free(To);
    ATO_IGNORE(ctx);
}

#define STM_TS_FMT "20%02d-%02d-%02dT%02d:%02d:%02dZ"
#define STM_TS_FMT_MILLIS "20%02d-%02d-%02dT%02d:%02d:%02d.%sZ"
static void _build_xml(ato_Ctx *ctx, ato_Stm *stm, ato_Xml *xml, const char *endpointurl, ato_String *certificate)
{
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    static const char *function = "_build_xml";
    char buffer[100];
    bool ok = TRUE;
    ato_String *b64certificate = NULL;
    ato_Log *log = ato_ctx_log(ctx);
    int errcode = ATO_ERR_OK;

    assert(stm != NULL);
    assert(endpointurl != NULL);
    assert(certificate != NULL);

    if (_loglevel >= ATO_LOG_DEBUG)
        _printstuff(ctx, log, xml);

    ok = ato_xml_setnodevalue(xml, NULL, AddressName, endpointurl);

    errcode = ato_base64encode(ctx, &b64certificate, certificate);
    if (errcode != ATO_ERR_OK)
        // there will need to be an ATO_ERR_B64 thrown inside the ato_base64encode
        Throw ATO_CTX_NEWERR(ctx, ATO_STM_ERR_GENERAL, "Could not base64 encode the certificate");

    ok = ato_xml_setnodevaluelen(xml, NULL, BinarySecurityTokenName, ato_str_value(b64certificate), ato_str_len(b64certificate));
    ato_str_free(b64certificate);
    if (!ok)
        // there will need to be an ATO_ERR_XML thrown inside the ato_base64encode
        Throw ATO_CTX_NEWERR(ctx, ATO_STM_ERR_GENERAL, "Could not set BinarySecurityTokenName node value to the base64 certificate");

    {
        char guid[50];
        strcpy(buffer, "urn:uuid:");
        strcat(buffer, ato_new_pseudo_guid(ctx, guid));
        ok = ato_xml_setnodevalue(xml, NULL, MessageIDName, buffer);

        ok = ato_xml_setnodevalue(xml, NULL, ToName, stm->url);
    }
    {
        time_t h_now; //Header
        struct tm h_tms, *h_t;
        time_t b_now; // Body
        struct tm b_tms, *b_t;

        h_t = &h_tms; b_t = &b_tms;
        time(&h_now); time(&b_now);
        ato_gmtime(&h_now, h_t); ato_gmtime(&b_now, b_t);
        h_t->tm_isdst = b_t->tm_isdst = -1;

        //Header
        h_t->tm_min += (int)stm->headermin_adj; mktime(h_t);
        sprintf(buffer, STM_TS_FMT_MILLIS, h_t->tm_year-100, h_t->tm_mon+1, h_t->tm_mday, h_t->tm_hour, h_t->tm_min, h_t->tm_sec, "001");
        ok = ato_xml_setnodevalue(xml, NULL, CreatedHdrName, buffer);

        h_t->tm_min += (int)stm->headermin_ttl; mktime(h_t);
        sprintf(buffer, STM_TS_FMT_MILLIS, h_t->tm_year-100, h_t->tm_mon+1, h_t->tm_mday, h_t->tm_hour, h_t->tm_min, h_t->tm_sec, "001");
        ok = ato_xml_setnodevalue(xml, NULL, ExpiresHdrName, buffer);

        //Body
        b_t->tm_min += (int)stm->bodymin_adj; mktime(b_t);
        sprintf(buffer, STM_TS_FMT, b_t->tm_year-100, b_t->tm_mon+1, b_t->tm_mday, b_t->tm_hour, b_t->tm_min, b_t->tm_sec);
        ok = ato_xml_setnodevalue(xml, NULL, CreatedBodyName, buffer);

        b_t->tm_min += (int)stm->bodymin_ttl; mktime(b_t);
        sprintf(buffer, STM_TS_FMT, b_t->tm_year-100, b_t->tm_mon+1, b_t->tm_mday, b_t->tm_hour, b_t->tm_min, b_t->tm_sec);
        ok = ato_xml_setnodevalue(xml, NULL, ExpiresBodyName, buffer);

//        sprintf(buffer, STM_TS_FMT_MILLIS, t->tm_year-100, t->tm_mon+1, t->tm_mday, t->tm_hour, t->tm_min, t->tm_sec, "001");
//        ok = ato_xml_setnodevalue(xml, NULL, ExpiresHdrName, buffer);
    }

    if (_loglevel >= ATO_LOG_DEBUG)
        _printstuff(ctx, log, xml);
}

//static void _dump_vars(const char *prefix, ato_Stm *stm) {
//    if (stm->vars != NULL) {
//      ato_ListNode *curr = NULL;
//      while ((curr = ato_lst_next(stm->vars, curr)) != NULL) {
//            printf("%s: %s=%s\n", prefix, ato_lst_key(curr), ato_lst_value(curr));
//      }
//    }
//}

#define _VAR_NAME_LEN   60
#define _VAR_NAME_TKN_S "@{"
#define _VAR_NAME_TKN_E "}"
// Extract out @{..} and put into var_name, then return start location in src
static const char *_var_getname(const char *src, char *var_name) {
    const char *src_start = strstr(src, _VAR_NAME_TKN_S);
    const char *src_end = NULL;
    size_t var_name_len = 0;
    *var_name = '\0';
    if (src_start == NULL) { return NULL; }
    src_end = strstr(src_start, _VAR_NAME_TKN_E);
    if (src_end == NULL) { return NULL; }
    var_name_len = src_end - src_start;
    if (var_name_len <= 0) { return NULL;}
    strncpy(var_name, src_start, var_name_len+1);
    var_name[var_name_len+1] = '\0';
    return src_start;
}

// with @{222}=world,@{111}=hello (@{333} doesn't exist in list)
// 12@{111}34@{222}56@{333}78 gets converted to 12hello34world56@{333}78
static ato_String *_rebuild_template(ato_Stm *stm, ato_String *xmltemplate) {
    const char *src = ato_str_value(xmltemplate);
    char *new_templatech = calloc(ato_str_len(xmltemplate) + stm->vars_total_length + 100, sizeof(char));
    char *dst = new_templatech;
    ato_String *new_template = NULL;
    char var_name[_VAR_NAME_LEN + 10];
    const char *src_start = NULL;
    //_dump_vars("_rebuild_template", stm);
    // For each varname, copy across the value or the varname
    while ((src_start = _var_getname(src, var_name)) != NULL) {
        const char *value = NULL;
        ato_ListNode *node = NULL;
        size_t len = 0;
        node = ato_lst_find(stm->vars, var_name);
        if (node != NULL) { value = ato_lst_value(node); }
        // if no value, just copy across the name
        if (value == NULL) { value = var_name; }
        len = src_start - src;
        strncpy(dst, src, len);
        strcpy(dst + len, value);
        dst += strlen(dst);
        src = src_start + strlen(var_name);
    }
    strcpy(dst, src);
    return ato_str_create(&new_template, new_templatech, ato_strlen(new_templatech), TRUE);
}

static bool _add_template_id_var(ato_Ctx *ctx, ato_Stm *stm) {
    char guid[50];
    const char *vars[] = {
        "id_0", ato_new_pseudo_guid(ctx, guid),
        NULL, NULL
    };
    return ato_stm_set_template_vars(ctx, stm, vars) == ATO_ERR_OK;
}

static ato_Xml *_loadtemplate(ato_Ctx *ctx, ato_Stm *stm, ato_String *xmltemplate, const char *endpointurl, ato_String *certificate)
{
    static const char *function = "_loadtemplate";
    ato_Xml *xml = NULL;
    ato_Log *log = ato_ctx_log(ctx);
    assert(_add_template_id_var(ctx, stm));
    if (stm->vars != NULL) {
        // Might be more efficient to patch the XML values??
        ato_String *t = _rebuild_template(stm, xmltemplate);
        if (_logdumpfile && ato_filewrite(t, "template.xml") != ATO_ERR_OK) {
            ATO_LOG_MSG(log,  ATO_LOG_ERR, "Failed to save file template.xml");
        }
        ato_xml_create(ctx, &xml, t);
        if (t != xmltemplate)
        ato_str_free(t);
    } else {
        ato_xml_create(ctx, &xml, xmltemplate);
    }

    _setns(ctx, xml);
    _build_xml(ctx, stm, xml, endpointurl, certificate);
    return xml;
}

/*
The template contains custom IDs rather than using the standard xml:id. Unfortunately these cannot
be changed since the service requires them.
There is an embedded DTD which tells the signing engine the IDs to use for lookups.
Note that once signing is complete, the DTD needs to be dropped before transmission.
*/
static void _sign(ato_Ctx *ctx, ato_Xml *xml, ato_String *privatekey)
{
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    static const char *function = "_sign";
    int ret = 0;
    xmlSecKeyPtr key = NULL;
    xmlNodePtr cur = NULL;
    xmlSecDSigCtx dsigCtx;
    xmlSecKeysMngrPtr gKeysMngr = NULL;
    int errcode = ATO_ERR_OK;

    Try {
        gKeysMngr = xmlSecKeysMngrCreate();
        if(gKeysMngr == NULL)
            Throw ATO_CTX_NEWERR(ctx, ATO_STM_ERR_GENERAL, "Could not create xmlSecKeysMngrPtr");

        if(xmlSecCryptoAppDefaultKeysMngrInit(gKeysMngr) < 0)
            Throw ATO_CTX_NEWERR(ctx, ATO_STM_ERR_GENERAL, "Could not init default keys manager");

        key = xmlSecCryptoAppKeyLoadMemory((const unsigned char *)ato_str_value(privatekey), ato_str_len(privatekey), xmlSecKeyDataFormatDer, NULL, NULL, NULL);
        if (key == NULL)
            Throw ATO_CTX_NEWERR(ctx, ATO_STM_ERR_GENERAL, "Could not load key into memory");

        ret = xmlSecCryptoAppDefaultKeysMngrAdoptKey(gKeysMngr, key);
        if (ret != 0)
            Throw ATO_CTX_NEWERR(ctx, ATO_STM_ERR_GENERAL, "Could not adopt key into default key manager");

        if (xmlSecDSigCtxInitialize(&dsigCtx, gKeysMngr) < 0)
            Throw ATO_CTX_NEWERR(ctx, ATO_STM_ERR_GENERAL, "Could not initialise dsig");

        cur = ato_xml_findnode(xml, NULL, NULL, NULL);
        if (cur == NULL)
            Throw ATO_CTX_NEWERR(ctx, ATO_STM_ERR_GENERAL, "Could not find xml node");

        cur = xmlSecFindNode(cur, xmlSecNodeSignature, xmlSecDSigNs);
        if(xmlSecDSigCtxSign(&dsigCtx, cur) < 0)
            Throw ATO_CTX_NEWERR(ctx, ATO_STM_ERR_GENERAL, "Could not sign xml node");

    } Catch (errcode) {
        goto finally;
    }

finally:
    if (gKeysMngr)
        xmlSecKeysMngrDestroy(gKeysMngr);
    xmlSecDSigCtxFinalize(&dsigCtx);
    if (errcode != ATO_ERR_OK) Throw errcode;
}

static void _decrypt(ato_Ctx *ctx, const char* enc_file, const char* key_file, const char* dec_file) {
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    static const char *function = "_decrypt";
    xmlDocPtr doc = NULL;
    xmlSecKeyPtr key = NULL;
    xmlNodePtr cur = NULL;
    xmlSecKeysMngrPtr gKeysMngr = NULL;
    xmlSecEncCtx encCtx;
    int errcode = ATO_ERR_OK;

    Try {
        gKeysMngr = xmlSecKeysMngrCreate();
        if(gKeysMngr == NULL)
            Throw ATO_CTX_NEWERR(ctx, ATO_STM_ERR_GENERAL, "Could not create xmlSecKeysMngrPtr");

        if(xmlSecCryptoAppDefaultKeysMngrInit(gKeysMngr) < 0)
            Throw ATO_CTX_NEWERR(ctx, ATO_STM_ERR_GENERAL, "Could not init default keys manager");

        key = xmlSecCryptoAppKeyLoad(key_file, xmlSecKeyDataFormatPkcs8Pem, NULL, NULL, NULL);
        if (key == NULL)
            Throw ATO_CTX_NEWERR(ctx, ATO_STM_ERR_GENERAL, "Could not load key from file");

        if (xmlSecCryptoAppDefaultKeysMngrAdoptKey(gKeysMngr, key) != 0)
            Throw ATO_CTX_NEWERR(ctx, ATO_STM_ERR_GENERAL, "Could not adopt key into default key manager");

        if (xmlSecEncCtxInitialize(&encCtx, gKeysMngr) < 0)
            Throw ATO_CTX_NEWERR(ctx, ATO_STM_ERR_GENERAL, "Could not initialise ctx");

        doc = xmlParseFile(enc_file);
        if (doc == NULL)
            Throw ATO_CTX_NEWERR(ctx, ATO_STM_ERR_GENERAL, "Could not load xml doc");

        cur = xmlDocGetRootElement(doc);
        if (cur == NULL)
            Throw ATO_CTX_NEWERR(ctx, ATO_STM_ERR_GENERAL, "Could not find xml root node");

        cur = xmlSecFindNode(cur, xmlSecNodeEncryptedData, xmlSecEncNs);
        if (cur == NULL)
            Throw ATO_CTX_NEWERR(ctx, ATO_STM_ERR_GENERAL, "Could not find xml node");

        if(xmlSecEncCtxDecrypt(&encCtx, cur) < 0)
            Throw ATO_CTX_NEWERR(ctx, ATO_STM_ERR_GENERAL, "Could not decrypt xml node");

        {
        xmlChar *buf = NULL;
        int size = 0;
        FILE *fp = fopen(dec_file, "wb");

        //fprintf(stdout, "Decrypted XML data:\n");
        xmlDocDumpMemory(doc, &buf, &size);
        fwrite(buf, sizeof(char), size, fp);
        if (fp != NULL) fclose(fp);
        if (buf != NULL) xmlFree(buf);
        }


    } Catch (errcode) {
        goto finally;
    }

finally:
    if (gKeysMngr)
        xmlSecKeysMngrDestroy(gKeysMngr);
    //xmlSecEncCtxDestroy(&encCtx);
    xmlSecEncCtxFinalize(&encCtx);
    if(doc != NULL) xmlFreeDoc(doc);
    if (errcode != ATO_ERR_OK) Throw errcode;
}

static bool _fileexists(const char *path)
{
    FILE *fp = NULL;
    if (path == NULL) { return FALSE; }
    fp = fopen(path, "r");
    if (fp != NULL) {
        fclose(fp);
    }
    return fp != NULL;
}

static void _dumpxml(ato_Ctx *ctx, ato_Log *log, ato_Xml *xml, const char *msg, const char *filename) {
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    const char *function = "_dumpxml";
    int errcode = ATO_ERR_OK;

    Try {
        if (_logdumpfile) {
            ATO_LOG_MSG(log, ATO_LOG_DEBUG, msg);
            ato_xml_savef(ctx, xml, filename);
        }
    } Catch(errcode) {
    }

    if (errcode != ATO_ERR_OK) Throw errcode;
}

static ato_String *_generate_rst(ato_Ctx *ctx, ato_Xml *xml, ato_String *privatekey) {
    ato_String *requestST = NULL;
    ato_Log *log = NULL;
    log = ato_ctx_log(ctx);

    assert(xml != NULL);
    assert(privatekey != NULL);

        _dumpxml(ctx, log, xml,  "Writing rst-presign.xml", "rst-presign.xml");
        _sign(ctx, xml, privatekey);
        ato_xml_dropdtd(xml);
        ato_xml_save(ctx, xml, &requestST);
    return requestST;
}

static ato_Net *_setnet(ato_Ctx *ctx, ato_Stm *stm)
{
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    static const char *function = "_setnet";
    ato_Net *net = NULL;
    ato_Log *log = ato_ctx_log(ctx);

    if (stm->url == NULL)
        Throw ATO_CTX_NEWERR(ctx, ATO_STM_ERR_GENERAL, "STS URL cannot be NULL");

    ATO_LOG_MSGV(log, ATO_LOG_DEBUG, "URL = %s", stm->url);

    ato_bnet_create(ctx, &net, stm->url, stm->contenttype);
    ato_bnet_settimeout(net, stm->timeout);
    ato_bnet_settimeoutconnect(net, stm->timeoutconnect);
    return net;
}

static ato_StmTkn *_request_stoken(ato_Ctx *ctx, ato_Stm *stm, ato_Xml *xml, ato_String *privatekey)
{
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    static const char *function = "_request_stoken";
    ato_NetData *response = NULL;
    ato_String *requestST = NULL, *requestSTresponse = NULL;
    ato_StmTkn *st = NULL;
    int errcode = ATO_ERR_OK;
    ato_Log *log = NULL;
    const char *rstxml = "rst.xml";
    const char *rstrxml = "rstr.xml";
    const char *rstrdxml = "rstr-decrypted.xml";
    ato_Net *net = _setnet(ctx, stm);
    log = ato_ctx_log(ctx);

    assert(xml != NULL);
    assert(privatekey != NULL);

    Try {

        requestST = _generate_rst(ctx, xml, privatekey);

        if (_logdumpfile && ato_filewrite(requestST, rstxml) != ATO_ERR_OK) {
            ATO_LOG_MSGV(log,  ATO_LOG_ERR, "Failed to save file %s", rstxml);
        }

        ato_bnet_settimeout(net, stm->timeout);
        ato_bnet_settimeoutconnect(net, stm->timeoutconnect);
        response = ato_bnet_request(ctx, net, requestST);
        requestSTresponse = ato_bnetd_content(ato_bnetd_item_primary(response));
        assert(requestSTresponse);

        ATO_LOG_MSGV(log, ATO_LOG_INFO, "%s", "Request SecurityToken Response received");
        if (_logdumpfile) {
            ATO_LOG_MSGV(log, ATO_LOG_DEBUG, "Writing response to file %s", rstrxml);
            if (ato_filewrite(requestSTresponse, rstrxml) != ATO_ERR_OK)
                ATO_LOG_MSGV(log,  ATO_LOG_ERR, "Failed to save file %s", rstrxml);
            else if (_fileexists(stm->rp_p8)) {
                _decrypt(ctx, rstrxml, stm->rp_p8, rstrdxml);
            }
        }

        ato__stmtkn_create(ctx, &st, requestSTresponse, "1");
    } Catch (errcode) {
    }

    ato_bnet_free(net);

    ato_str_free(requestST);
    if (errcode != ATO_ERR_OK) Throw errcode;
    return st;
}

static void _stm_set_template_var(ato_Ctx *ctx, ato_Stm *stm, const char *name, const char *value) {
    char var_name[_VAR_NAME_LEN + 10];
    ato_ListNode *node = NULL;
    strcpy(var_name, _VAR_NAME_TKN_S);
    strcat(var_name, name);
    strcat(var_name, _VAR_NAME_TKN_E);
    node = ato_lst_find(stm->vars, var_name);
    if (node != NULL) {
        char *v = ato_lst_value(node);
        if (v != NULL) { stm->vars_total_length -= strlen(v); ato_free(v); }
        ato_lst_setvalue(node, ato_strdup(value, 0));
    } else {
        ato_lst_add(stm->vars, var_name, ato_strdup(value, 0));
    }
    stm->vars_total_length += strlen(value);
    ATO_IGNORE(ctx);
}

static void _stm_set_template_vars(ato_Ctx *ctx, ato_Stm *stm, const char * const *vars) {
    int i = 0;
    const char *name = NULL, *value = NULL;
    if (vars == NULL) { _ato_free_vars(stm); return; }
    if (vars[i] == NULL || vars[i+1] == NULL || stm->xtemplate == NULL) { return; }
    if (stm->vars == NULL) { ato_lst_create(&(stm->vars)); }
    name = vars[i++];
    value = vars[i++];
    while (name != NULL && value != NULL) {
        _stm_set_template_var(ctx, stm, name, value);
        name = vars[i++];
        value = vars[i++];
    }
}

static void _stm_set_template_varstr(ato_Ctx *ctx, ato_Stm *stm, const char *varstr) {
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    static const char *function = "_stm_set_template_varstr";
    int errcode = ATO_ERR_OK;
    char **vars = NULL;
    ATO_CTX_FN_START(ctx);
    //printf("_stm_set_template_varstr\n");
    Try {
        if (varstr != NULL) {
            ato_strtok2array(&vars, varstr, ",");
        }
        _stm_set_template_vars(ctx, stm, (const char * const *) vars);
    } Catch(errcode) {
    }
    ato_strtokfree(vars);
    ATO_RETHROW_ONERR(errcode);
    ATO_CTX_FN_END(ctx, errcode);
}
/*********************************************************************************/
static void _setloglevel(ato_eLoglevel level)
{
    _loglevel = level;
}
static void _setlogdumpfile(bool logdumpfile)
{
    _logdumpfile = logdumpfile;
}
/*********************************************************************************/
int ato__stm_init(void)
{
    static bool invoked = FALSE;
    if (invoked) return ATO_ERR_OK;
    invoked = TRUE;

    {
        ato_LibModule *lm = ato_initfnloglevel(_library, _module, _moduleid, _loglevel, _setloglevel);
        ato_initfnlogfiledump(lm, _logdumpfile, _setlogdumpfile);
    }
    return ATO_ERR_OK;
}

void ato__stm_deinit(void)
{
}

static void check_n_set_time(ato_Ctx *ctx, const char *vname, int *time_var) {
    static const char *function = "ato__stm_create";
    ato_Log *log = ato_ctx_log(ctx);
    const char *v = ato_ctx_cachec(ctx, vname);
    if (v != NULL && strlen(v) <= 7) {
        *time_var = strtol(v, NULL, 10);
        ATO_LOG_MSGV(log, ATO_LOG_DEBUG, "Using %s = %d minutes", vname, *time_var);
    }
}
/*********************************************************************************/
void ato__stm_create(ato_Ctx *ctx, ato_Stm **obj, const char *stsurl, const char *stmtemplate)
{
    static const char *function = "ato__stm_create";
    static const char *contenttype = "application/soap+xml; charset=utf-8";
    int errcode = ATO_ERR_OK;
    ato_Stm *stm = NULL;
    ato_Log *log = ato_ctx_log(ctx);

    ATO_CTX_FN_START(ctx);

    ATO_ASSERT_ISNOTALLOCATED(obj);

    *obj = stm = calloc(1, sizeof(ato_Stm));
    assert(stm != NULL);

    stm->timeout = ato_bnet_timeout_default(ctx);
    stm->timeoutconnect = ato_bnet_timeoutconnect_default(ctx);

    stm->headermin_adj = stm->bodymin_adj = 0;
    stm->headermin_ttl = stm->bodymin_ttl = ATO_STM_EXPIRY_MINS_DEFAULT;
    
    stm->contenttype = ato_strdup(contenttype, 0);

    stm->url = ato_strdup(stsurl, 0);

    ato_str_create(&(stm->xtemplate), ato_strdup(stmtemplate, 0), ato_strlen(stmtemplate), TRUE);


    {
        const char *v = ato_ctx_cachec(ctx, "stm.rp_p8");
        stm->rp_p8 = ato_strdup(v, 0);
        ATO_LOG_MSGV(log, ATO_LOG_DEBUG, "Using %s = %s", "stm.rp_p8", v == NULL ? "" : v);
        check_n_set_time(ctx, "stm.headermin_adj", &(stm->headermin_adj));
        check_n_set_time(ctx, "stm.bodymin_adj", &(stm->bodymin_adj));
        check_n_set_time(ctx, "stm.headermin_ttl", &(stm->headermin_ttl));
        check_n_set_time(ctx, "stm.bodymin_ttl", &(stm->bodymin_ttl));
    }

    ATO_CTX_FN_END(ctx, errcode);
}


void ato__stm_stsissue(ato_Ctx *ctx, ato_Stm *stm, ato_StmTkn **st, ato_String *certificate, ato_String *privatekey, const char *relyingpartyurl)
{
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    static const char *function = "ato__stm_stsissue";
    int errcode = ATO_ERR_OK;
    ato_Xml *xml = NULL;

    ATO_CTX_FN_START(ctx);

    ATO_ASSERT_ISNOTALLOCATED(st);

    if (relyingpartyurl == NULL) {
        Throw ATO_CTX_NEWERR(ctx, ATO_STM_ERR_GENERAL, "Relying party cannot be NULL");
    }

    Try {
        xml = _loadtemplate(ctx, stm, stm->xtemplate, relyingpartyurl, certificate);
        *st = _request_stoken(ctx, stm, xml, privatekey);
    } Catch(errcode) {
    }

    ato_xml_free(xml);
    ATO_RETHROW_ONERR(errcode);
    ATO_CTX_FN_END(ctx, errcode);
}

void ato__stm_loadst(ato_Ctx *ctx, ato_Stm *stm, ato_StmTkn **st, ato_String *atorstr)
{
    static const char *function = "ato__stm_loadst";
    int errcode = ATO_ERR_OK;

    ATO_IGNORE(stm);
    ATO_CTX_FN_START(ctx);
    ato__stmtkn_create(ctx, st, atorstr, NULL);
    ATO_CTX_FN_END(ctx, errcode);
}

/*********************************************************************************/
int ato_stm_create(ato_Ctx *ctx, ato_Stm **stm, const char *stsurl, const char *stmtemplate)
{
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    static const char *function = "ato_stm_create";
    int errcode = ATO_ERR_OK;
    ATO_CTX_FN_START(ctx);
    Try {
        ato__stm_create(ctx, stm, stsurl, stmtemplate);
    } Catch (errcode) {
        errcode = ato__stmlib_process_errcode(ctx, errcode, _module, function, __LINE__);
    }
    ATO_CTX_FN_END(ctx, errcode);
    return errcode;
}

void ato_stm_purgecache(ato_Ctx *ctx, bool purgeall)
{
    ATO_IGNORE(ctx);
    ATO_IGNORE(purgeall);
}

void ato_stm_free(ato_Ctx *ctx, ato_Stm *stm)
{
    _ato_free(stm);
    ato_stm_purgecache(ctx, TRUE);
}

void ato_stm_setexpiryinterval(ato_Stm *stm, size_t expirymins)
{
    size_t emins = expirymins;

    if (emins < ATO_STM_EXPIRY_MINS_MIN)
        emins = ATO_STM_EXPIRY_MINS_MIN;
    if (emins > ATO_STM_EXPIRY_MINS_MAX)
        emins = ATO_STM_EXPIRY_MINS_MAX;
    stm->bodymin_ttl = (int) emins;
}

size_t ato_stm_expiryinterval(ato_Stm *stm)
{
    return stm->bodymin_ttl;
}

int ato_stm_set_template_vars(ato_Ctx *ctx, ato_Stm *stm, const char * const *vars) {
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    static const char *function = "ato_stm_set_template_vars";
    int errcode = ATO_ERR_OK;
    ATO_CTX_FN_START(ctx);
    Try {
        _stm_set_template_vars(ctx, stm, vars);
    } Catch (errcode) {
        errcode = ato__stmlib_process_errcode(ctx, errcode, _module, function, __LINE__);
    }
    ATO_CTX_FN_END(ctx, errcode);
    return errcode;
}
int ato_stm_set_template_varstr(ato_Ctx *ctx, ato_Stm *stm, const char *varstr) {
        struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    static const char *function = "ato_stm_set_template_varstr";
    int errcode = ATO_ERR_OK;
    ATO_CTX_FN_START(ctx);
    Try {
        _stm_set_template_varstr(ctx, stm, varstr);
    } Catch (errcode) {
        errcode = ato__stmlib_process_errcode(ctx, errcode, _module, function, __LINE__);
    }
    ATO_CTX_FN_END(ctx, errcode);
    return errcode;
}

int ato_stm_stsissue(ato_Ctx *ctx, ato_Stm *stm, ato_StmTkn **st, ato_String *certificate, ato_String *privatekey, const char *relyingpartyurl)
{
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    static const char *function = "ato_stm_stsissue";
    int errcode = ATO_ERR_OK;
    ATO_CTX_FN_START(ctx);
    Try {
        ato__stm_stsissue(ctx, stm, st, certificate, privatekey, relyingpartyurl);
    } Catch (errcode) {
        errcode = ato__stmlib_process_errcode(ctx, errcode, _module, function, __LINE__);
    }
    ATO_CTX_FN_END(ctx, errcode);
    return errcode;
}

int ato_stm_loadst(ato_Ctx *ctx, ato_Stm *stm, ato_StmTkn **st, ato_String *atorstr)
{
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    static const char *function = "ato_stm_loadst";
    int errcode = ATO_ERR_OK;
    ATO_CTX_FN_START(ctx);
    Try {
        ato__stm_loadst(ctx, stm, st, atorstr);
    } Catch (errcode) {
        errcode = ato__stmlib_process_errcode(ctx, errcode, _module, function, __LINE__);
    }
    ATO_CTX_FN_END(ctx, errcode);
    return errcode;
}

size_t ato_stm_timeout(ato_Ctx *ctx, ato_Stm *stm)
{
    ATO_IGNORE(ctx);
    return stm->timeout;
}

void ato_stm_settimeout(ato_Ctx *ctx, ato_Stm *stm, size_t timeout)
{
    ATO_IGNORE(ctx);
    stm->timeout = timeout;
}

size_t ato_stm_timeoutconnect(ato_Ctx *ctx, ato_Stm *stm)
{
    ATO_IGNORE(ctx);
    return stm->timeoutconnect;
}

void ato_stm_settimeoutconnect(ato_Ctx *ctx, ato_Stm *stm, size_t timeout)
{
    ATO_IGNORE(ctx);
    stm->timeoutconnect = timeout;
}
