/*
This file is for documentation purposes only.
*/

//-----------------------------------------------------------

/*! @addtogroup sbrcsr_gettingstarted_error

@section sbrcsr_gettingstarted_error_overview Overview

SBRCSR errors originate from 3 broad sources:
- "normal" library related errors of various kinds
- soap errors from network related activities
- business document errors where a Response document has been received

The first two are returned to the client code as errcodes and ato_Error objects. The last is not
viewed as an API error but a business processing error and should be handling as per the SBR WIG
(Web services Implementation Guide).

When submitting documents to SBR, first check if an errcode has been returned, and if so process the error as per
normal error handling further below.

@section sbrcsr_gettingstarted_error_business Business Document Errors

If no API error has occurred, the returned response should be checked for business level errors. This involves
checking the Response eventitems elements for information. In particular,
sbr_res_ismaxseveritycode("Error") and sbr_res_ismaxseveritycode("Warning") should be used to check for problems.

Following is a code snippet reflecting the approach outlined in the WIG.
See the WIG and sample code for further details.

@code

static void _process_eventitems(sbr_Response *response)
{
    size_t indent = 1;
    sbr_Sbdm *sbdm = NULL;
    size_t i = 1, icount = 0;

    icount = sbr_res_sbdm_count(response);
    for (sbdm = sbr_res_sbdm_firstv(response); sbdm; sbdm = sbr_res_sbdm_nextv(response), i++) {
        sbr_MsgEventItem *ei = NULL;
        size_t j = 1, jcount = sbr_sbdm_eventitem_count(sbdm);
        printf("SBDM %d of %d\n", i, icount);
        for (ei = sbr_sbdm_eventitem_firstv(sbdm); ei; ei = sbr_sbdm_eventitem_nextv(sbdm), j++) {
            dump_eventitem(stdout, &indent, ei, j, jcount, FALSE);
        }
    }
}

static void _process_csrresponse(ato_Ctx *ctx, sbr_Response *response)
{
    if (sbr_res_ismaxseveritycode(response, "Error")) {
        printf("maxseveritycode of %s found\n", "Error");
        _process_eventitems(response);
    } else {
        if (sbr_res_ismaxseveritycode(response, "Warning")) {
            printf("maxseveritycode of %s found\n", "Warning");
        } else if (sbr_res_ismaxseveritycode(response, "Information")) {
            printf("maxseveritycode of %s found\n", "Information");
        } else {
            printf("maxseveritycode of %s found\n", "?unknown?");
        }
        _process_eventitems(response);
        dump_response(ctx, response, "sbrcsr_response_dump.txt");
    }
}

@endcode



@section sbrcsr_gettingstarted_error_handling Handling API Errors

SBRCSR uses the error handling mechanism in ATOBASE.

For iterating errors see @ref atobase_error_iterating.

SBRCSR API methods which return errcodes will always return an SBR_CSR_ERR_.. code. For details on
errcodes see @ref sbrcsr_api_csr_types_errcodes.

Top level errcodes can be checked using:

@code
    switch (errcode) {
        case SBR_CSR_ERR_GENERAL:
            printf("CSR General error: ");
            break;
        case SBR_CSR_ERR_NETSENDER:
            printf("CSR Sender error: ");
            break;
        case SBR_CSR_ERR_NETRECEIVER:
            printf("CSR Receiver error: ");
            break;
        case SBR_CSR_ERR_NETUNAVAILABLE:
            printf("CSR Remote Service not available: ");
            break;
        case SBR_CSR_ERR_NETCOMMS:
            printf("CSR Network error: ");
            break;
        case SBR_CSR_ERR_NETTIMEOUT:
            printf("CSR Remote Service timeout: ");
            break;
    }
@endcode

*/
