#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>
#include <time.h>
#include <stdarg.h>

#include "atobase/private/pall.h"
#include "atobnet/private/pall.h"

#include "atointernal.h"
#include "inttypes.h"
#include "intmsgevent.h"
#include "intmsgts.h"
#include "intmsgparty.h"
#include "intmsgrec.h"
#include "intdoca.h"
#include "intdoc.h"
#include "intsbdm.h"
#include "intresponse.h"

static const char *_library = SBR_CSR_LIBRARY;
static const char *_module = SBR_CSR_MODULE_RESPONSE;
static unsigned long _moduleid = SBR_CSR_MODULEID_RESPONSE;

static ato_eLoglevel _loglevel = ATO_LOG_WARN;
static bool _logdumpfile = FALSE;

static const char *xServiceName = "/s:Envelope/s:Body/service:$";

/*********************************************************************************/

struct _sbr_Response {
    sbr_eMsgSourceType msgsrctype;
    char *servicename;
    void *xservicenode;
    ato_Xml *xml;
    ato_List *sbdms;
    ato_Iterator *iter;
};


/*********************************************************************************/
static void _ato_free(sbr_Response *response)
{
    if (response == NULL) return;

    if (response->servicename != NULL) free(response->servicename);
    if (response->sbdms != NULL) {
        ato_ListNode *node = NULL;
        while ((node = ato_lst_next(response->sbdms, node)) != NULL) {
            sbr__sbdm_free(ato_lst_value(node));
        }
        ato_lst_free(response->sbdms);
    }
    ato_xml_free(response->xml);
    ato_iter_free(response->iter);
}

static void _process_response(ato_Ctx *ctx, sbr_Response *response)
{
    static const char *xSBDMs = "//data:StandardBusinessDocumentMessage";
    void *xlist = NULL;
    size_t i = 0, count = 0;

    xlist = ato_xml_listload(response->xml, NULL, xSBDMs, NULL);
    if (xlist == NULL) return;

    count = ato_xml_listcount(response->xml, xlist);
    for (i = 0; i < count; i++) {
        sbr_Sbdm *sbdm = NULL;
        sbr__sbdm_create(ctx, &sbdm, response->msgsrctype, i, response->xml, response->xservicenode);
        ato_lst_add(response->sbdms, sbr_sbdm_iid(sbdm), sbdm);
    }
    ato_xml_listfree(response->xml, xlist);
}

static void _setns(ato_Ctx *ctx, ato_Xml *xml, const char *servicens)
{
    assert(xml != NULL); assert(servicens != NULL);
    ato_xml_addnamespace(ctx, xml, "i", "http://www.w3.org/2001/XMLSchema-instance");
    ato_xml_addnamespace(ctx, xml, "s", "http://www.w3.org/2003/05/soap-envelope");
    ato_xml_addnamespace(ctx, xml, "d4p1", "http://docs.oasis-open.org/wss/2004/01/oasis-200401-wss-wssecurity-utility-1.0.xsd");
    ato_xml_addnamespace(ctx, xml, "d2p1", "http://docs.oasis-open.org/wss/2004/01/oasis-200401-wss-wssecurity-utility-1.0.xsd");
    ato_xml_addnamespace(ctx, xml, "xenc", "http://www.w3.org/2001/04/xmlenc#");
    ato_xml_addnamespace(ctx, xml, "wsse", "http://docs.oasis-open.org/wss/2004/01/oasis-200401-wss-wssecurity-secext-1.0.xsd");

    ato_xml_addnamespace(ctx, xml, "sig", "http://www.w3.org/2000/09/xmldsig#");
    //ato__xml_setns(ctx, xml, "wst", "http://docs.oasis-open.org/ws-sx/ws-trust/200512");
    ato_xml_addnamespace(ctx, xml, "saml", "urn:oasis:names:tc:SAML:2.0:assertion");
    ato_xml_addnamespace(ctx, xml, "data", "http://sbr.gov.au/comn/sbdm.02.data");
    ato_xml_addnamespace(ctx, xml, "service", servicens); //"tns"

    ato_xml_addnamespace(ctx, xml, "err", "http://sbr.gov.au/comn/sbrfaults.200809");
    ato_xml_addnamespace(ctx, xml, "nsmime", "http://www.w3.org/2005/05/xmlmime");

    ato_xml_addnamespace(ctx, xml, "xop", "http://www.w3.org/2004/08/xop/include");
}

/*********************************************************************************/
static void _setloglevel(ato_eLoglevel level)
{
    _loglevel = level;
}
static void _setlogdumpfile(bool logdumpfile)
{
    _logdumpfile = logdumpfile;
}

/*********************************************************************************/
int sbr__res_init(void)
{
    static bool invoked = FALSE;
    if (invoked) return ATO_ERR_OK;
    invoked = TRUE;

    {
        ato_LibModule *lm = ato_initfnloglevel(_library, _module, _moduleid, _loglevel, _setloglevel);
        ato_initfnlogfiledump(lm, _logdumpfile, _setlogdumpfile);
    }

    return ATO_ERR_OK;
}

void sbr__res_deinit(void)
{
}

static void _dumpxml(ato_Ctx *ctx, ato_Log *log, ato_Xml *xml, const char *msg, const char *filename)
{
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    const char *function = "_dumpxml";
    int errcode = ATO_ERR_OK;

    Try {
        if (_logdumpfile) {
            ATO_LOG_MSG(log, ATO_LOG_DEBUG, msg);
            ato_xml_savef(ctx, xml, filename);
        }
    } Catch(errcode) {
    }

    if (errcode != ATO_ERR_OK) Throw errcode;
}

static void _loadxml(ato_Ctx *ctx, sbr_Response *response, ato_NetData *netdata, const char *servicenamespace)
{
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    const char *function = "_loadxml";
    int errcode = ATO_ERR_OK;
    ato_String *debugresponsebuf = NULL;
    ato_String *rstr = NULL;
    ato_Log *log = NULL;
    assert(ctx != NULL);
    log = ato_ctx_log(ctx);

    rstr = ato_bnetd_content(ato_bnetd_item_primary(netdata));
    assert(rstr != NULL);

    ATO_LOG_MSG(log, ATO_LOG_INFO, "=========== SBR Response received ===========");
    if (_logdumpfile) {
        ATO_LOG_MSG(log, ATO_LOG_INFO, "Writing response to file csr-response.xml");
        ato_filewrite(rstr, "csr-response.xml");
    }

#if defined(DEBUG) || defined(_DEBUG)
    ato_fileload(&debugresponsebuf, "debugresponse.xml");
#endif

    if (debugresponsebuf) {
        ATO_LOG_MSG(log, ATO_LOG_INFO, "Processing debugresponse.xml");
        Try {
            ato_xml_create(ctx, &(response->xml), debugresponsebuf);
        } Catch(errcode) {
        }
        ato_str_free(debugresponsebuf);
    } else {
        ato_xml_create(ctx, &(response->xml), rstr);
    }
    if (errcode != ATO_ERR_OK) Throw errcode;

    _setns(ctx, response->xml, servicenamespace);

    if (ato_bnetd_mtom_convertfrom(ctx, netdata, response->xml)) {
        _dumpxml(ctx, log, response->xml,  "Response altered - writing csr-response-mtom.xml", "csr-response-mtom.xml");
    }

}

/*********************************************************************************/
void sbr__res_create(ato_Ctx *ctx, sbr_Response **obj, ato_NetData *netdata, const char *servicename, const char *servicenamespace)
{
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    const char *function = "sbr__res_create";
    int errcode = ATO_ERR_OK;
    sbr_Response *response = NULL;
    assert(ctx != NULL);

    ATO_CTX_FN_START(ctx);

    ATO_ASSERT_ISNOTALLOCATED(obj);

    *obj = response = calloc(1, sizeof(sbr_Response));
    assert(response != NULL);

    _loadxml(ctx, response, netdata, servicenamespace);

    response->msgsrctype = SBR_RESPONSE;
    ato_lst_create(&(response->sbdms));
    response->servicename = ato_strdup("Response", strlen(servicename));
    strcat(response->servicename, servicename);

    response->xservicenode = ato_xml_findnode(response->xml, NULL, xServiceName, response->servicename);
    if (!response->xservicenode)
        Throw ATO_CTX_NEWERR(ctx, SBR_CSR_ERR_GENERAL, "Invalid response received");

    _process_response(ctx, response);

    response->iter = ato_iter_create_lst(response->sbdms);

    ATO_CTX_FN_END(ctx, errcode);
}

void sbr__res_free(sbr_Response *response)
{
    if (response == NULL) return;

    _ato_free(response);
    free(response);
}

void sbr__res_xml(ato_Ctx *ctx, sbr_Response *response, char **buffer)
{
    const char *function = "sbr__res_xml";
    ato_String *str = NULL;
    int errcode = ATO_ERR_OK;

    ATO_CTX_FN_START(ctx);
    ATO_ASSERT_ISNOTALLOCATED(buffer);

    *buffer = NULL;
    ato_xml_save(ctx, response->xml, &str);
    *buffer = ato_strdup(ato_str_value(str), 0);
    ato_str_free(str);

    ATO_CTX_FN_END(ctx, errcode);
}

/*********************************************************************************/
int sbr_res_xml(ato_Ctx *ctx, sbr_Response *response, char **buffer)
{
    struct exception_context *the_exception_context = ato__ctx_ec(ctx);
    static const char *function = "sbr_res_xml";
    int errcode = ATO_ERR_OK;
    ATO_CTX_FN_START(ctx);
    Try {
        sbr__res_xml(ctx, response, buffer);
    } Catch (errcode) {
        errcode = sbr__csrlib_process_errcode(ctx, errcode, _module, function, __LINE__);
    }
    ATO_CTX_FN_END(ctx, errcode);
    return errcode;
}

size_t sbr_res_sbdm_count(sbr_Response *response)
{
    assert(response != NULL);
    return ato_lst_count(response->sbdms);
}
void sbr_res_sbdm_iterator(sbr_Response *response, ato_Iterator **iter)
{
    ATO_ASSERT_ISNOTALLOCATED(iter);
    if (response == NULL) return;
    *iter = ato_iter_create_lst(response->sbdms);
}
sbr_Sbdm *sbr_res_sbdm_firstv(sbr_Response *response)
{
    if (response == NULL) return NULL;
    return ato_iter_firstv(response->iter);
}
sbr_Sbdm *sbr_res_sbdm_nextv(sbr_Response *response)
{
    if (response == NULL) return NULL;
    return ato_iter_nextv(response->iter);
}

bool sbr_res_ismaxseveritycode(sbr_Response *response, const char *severitycode)
{
    ato_ListNode *node = NULL;
    assert(response != NULL);
    while ((node = ato_lst_next(response->sbdms, node)) != NULL) {
        const char *msc = sbr_sbdm_maxseveritycode(ato_lst_value(node));
        if (ato_streqi(msc, severitycode))
            return TRUE;
    }
    return FALSE;
}
