#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <time.h>
#include <assert.h>
#include <stdarg.h>
#include <ctype.h>
#include <sys/stat.h>

#include <curl/curl.h>

#include "akversion.h"

#include "atobase/all.h"
#include "atobnet/all.h"
#include "atoakm/all.h"
#include "atostm/all.h"
#include "sbrcsr/all.h"

#include "dump.h"
#include "amend.h"

#include "../common/util.h"
#include "../common/log.h"

#include "sys.h"
#include "misc.h"

ato_Ctx *g_ctx = NULL;
ato_Log *g_log = NULL;
ato_eLoglevel arg_loglevel = ATO_LOG_ALL;

static const char *_DEFAULT_ksdir = "."; // current folder
static const char *_DEFAULT_kspwd = "Password1!";
static const char *_DEFAULT_ksalias = "ABRP:27809366375_10000022";
static const char *_DEFAULT_messagetype = "message.ping";
static const char *_DEFAULT_agency = "ato.gov.au";

//static const char *_DEFAULT_govgenid = NULL;
//static const char *_DEFAULT_busgenid = "028386B6-2E8A-4fbf-9FFC-845ECAE10FCA";

#if defined(WIN32) || defined(_WIN32) || defined(WIN64) || defined(_WIN64)
#else
#endif

static const char *arg_organisation = NULL;
static const char *arg_installprefix = NULL;
static bool arg_logfd = FALSE;
static const char *arg_logpath = NULL;
static const char *arg_netlogflag = NULL;
static const char *arg_proxyurl = NULL;
static const char *arg_proxyauthtype = NULL;
static const char *arg_proxybypass = NULL;
static const char *arg_proxyuser = NULL;
static const char *arg_proxypwd = NULL;
static const char *arg_proxypwdfile = NULL;
static const char *arg_sslversion = NULL;
static const char *arg_sslcainfo = NULL;

static const char *arg_ksdir = NULL;
static const char *arg_ksfile = NULL;
static const char *arg_kspwd = NULL;
static const char *arg_ksalias = NULL;
static const char *arg_chpwd = NULL;
static size_t arg_creds = 3;
static bool arg_crypto = FALSE;

static int arg_exptime = 0;
static size_t arg_nettimeout = 120;
static size_t arg_connecttimeout = 30;
static bool arg_forcestmerror = FALSE;
static const char *arg_securitytokenfile = NULL;
static const char *arg_stmvars = NULL;
static int arg_stmrepeat = 0;
static const char *arg_stmtemplate = NULL;
static const char *arg_stmservice = NULL;
static const char *arg_relyingparty = NULL;
static const char *arg_stm_rp_p8 = NULL;

static const char *arg_servicetype = NULL;
static const char *arg_messagetype = NULL;
static const char *arg_agency = NULL;
static bool arg_forcecsrerror = FALSE;
static size_t arg_mtomthreshold = 0;
static const char *arg_runtest = NULL;
static const char *arg_busdoclist = NULL;
static const char *arg_vurilist = NULL;
static const char *arg_govgenid = NULL;
static const char *arg_busgenid = NULL;
static const char *arg_ssid = NULL;
static bool arg_csrskip = FALSE;
static const char *arg_csrtemplate = NULL;
static const char *arg_csrservice = NULL;

static void _addargs(void)
{
        ato_arg_add("organisation", TRUE, NULL, NULL, "Name of the organisation or person running this");
        ato_arg_add("installprefix", FALSE, NULL, NULL, "Where to look for dependent library resources, if any");
        ato_arg_add("logfd", FALSE, NULL, NULL, "Some modules will dump data to files for debugging purposes");
        ato_arg_add("logpath", FALSE, "messages.log", NULL, "Set the log path");
        ato_arg_add("loglevel", FALSE, "d", "off|fatal|err|warn|info|debug|trace|all", "Set the loglevel - only need first letter");
        ato_arg_add("netlogflag", FALSE, NULL, NULL, "the network library has additional loglevel settings");
        ato_arg_add("proxyurl", FALSE, NULL, NULL, "proxy server url");
        ato_arg_add("proxyauthtype", FALSE, NULL, NULL, "proxy server authentication type: any (default); basic; digest; ntlm");
        ato_arg_add("proxybypass", FALSE, NULL, NULL, "proxy server bypass list");
        ato_arg_add("proxyuser", FALSE, NULL, NULL, "proxy server user name - if not supplied read the settings from the current config file context");
        ato_arg_add("proxypwd", FALSE, NULL, NULL, "proxy server user password");
        ato_arg_add("proxypwdfile", FALSE, NULL, NULL, "read proxyuser (1st line) and proxypwd (2nd line) from this file");
        ato_arg_add("sslversion", FALSE, NULL, NULL, "CURL options for SSL/TLS(DEFAULT,TLSv1,SSLv2,SSLv3,TLSv1_0,TLSv1_1,TLSv1_2)");
        ato_arg_add("sslcainfo", FALSE, NULL, NULL, "Path to CURL CA store or 'off' to diable SSL - see CURL option CURLOPT_CAINFO");

        ato_arg_add("ksdir", FALSE, ".", NULL, "Folder to look for the kestore");
        ato_arg_add("ksfile", FALSE, ato_akm_default_ksfile(), NULL, "Filename of the keystore");
        ato_arg_add("kspwd", FALSE, "Password1!", NULL, "Password for the keystore");
        ato_arg_add("ksalias", FALSE, NULL, NULL, "Alias to load from the keystore");
        ato_arg_add("chpwd", FALSE, NULL, NULL, "keystore password to change to");
        ato_arg_add("creds", FALSE, "3", NULL, "Number of credentials to list");
        ato_arg_add("crypto", FALSE, NULL, NULL, "Dump private key details");

        ato_arg_add("exptime", FALSE, NULL, NULL, "Expiry time for SecurityTokens");
        ato_arg_add("nettimeout", FALSE, "120", NULL, "Total seconds to wait for transaction to complete");
        ato_arg_add("connecttimeout", FALSE, "30", NULL, "Total seconds to wait for a connection");
        ato_arg_add("forcestmerror", FALSE, NULL, NULL, "Force an error in the SecurityToken request");
        ato_arg_add("securitytokenfile", FALSE, NULL, NULL, "security token file used to load token into cache");
        ato_arg_add("stmvars", FALSE, NULL, NULL, "Template vars to use");
        ato_arg_add("stmrepeat", FALSE, NULL, NULL, "Repeat call to STS n times");
        ato_arg_add("stmtemplate", FALSE, ato_stm_default_templatefile(), NULL, "STM template file");
        ato_arg_add("stmservice", FALSE, NULL, NULL, "URL for STS service");
        ato_arg_add("relyingparty", FALSE, NULL, NULL, "Relying Party");
        ato_arg_add("stm.rp_p8", FALSE, "rstr.p8.pem", NULL, "Relying Party decryption key to descrypt the STM RSTR");

        ato_arg_add("servicetype", FALSE, NULL, NULL, "Service type to use");
        ato_arg_add("messagetype", FALSE, "message.ping", NULL, "Message type to use");
        ato_arg_add("agency", FALSE, NULL, NULL, "Agency to use");
        ato_arg_add("forcecsrerror", FALSE, "0", "0|1", "Cause an invalid XML request to be submitted which will generated an XML error response");
        ato_arg_add("mtomthreshold", FALSE, "0", NULL, "size in bytes of documents before they are 'MTOMed'");
        ato_arg_add("runtest", FALSE, NULL, "2", "Will run extra tests");
        ato_arg_add("busdoclist", FALSE, NULL, NULL, "Comma separated list of business documents - see also args-def.run");
        ato_arg_add("vurilist", FALSE, NULL, NULL, "Comma separated list validation uri");
        ato_arg_add("govgenid", FALSE, NULL, NULL, "Government Identifier");
        ato_arg_add("busgenid", FALSE, NULL, NULL, "Business Identifier");
        ato_arg_add("ssid", FALSE, NULL, NULL, "softwareSubscriptionId");
        ato_arg_add("csrskip", FALSE, "0", "0|1", "Force the CSR sample to exit with success without running any tests");
        ato_arg_add("csrtemplate", FALSE, sbr_csr_default_templatefile(), NULL, "CSR template file");
        ato_arg_add("csrservice", FALSE, NULL, NULL, "URL for CSR service");
}

static void _initargs(void)
{
        arg_organisation = ato_arg_value("organisation");
        arg_installprefix = ato_arg_value("installprefix");;
        arg_logfd = ato_arg_value_bool("logfd");
        arg_logpath = ato_arg_value("logpath");
        arg_loglevel = ato_log_name2level(ato_arg_value("loglevel"));
        arg_netlogflag = ato_arg_value("netlogflag");
        arg_proxyurl = ato_arg_value("proxyurl");
        arg_proxyauthtype = ato_arg_value("proxyauthtype");
        arg_proxybypass = ato_arg_value("proxybypass");
        arg_proxyuser = ato_arg_value("proxyuser");
        arg_proxypwd = ato_arg_value("proxypwd");
        arg_proxypwdfile = ato_arg_value("proxypwdfile");
        arg_sslversion = ato_arg_value("sslversion");
        arg_sslcainfo = ato_arg_value("sslcainfo");

        arg_ksdir = ato_arg_value("ksdir");
        arg_ksfile = ato_arg_value("ksfile");
        arg_kspwd = ato_arg_value("kspwd");
        arg_ksalias = ato_arg_value("ksalias");
        arg_chpwd = ato_arg_value("chpwd");
        arg_creds = (size_t) ato_arg_value_int("creds");
        arg_crypto = ato_arg_value_bool("crypto");

        arg_exptime = ato_arg_value_int("exptime");
        arg_nettimeout = (size_t) ato_arg_value_int("nettimeout");
        arg_connecttimeout = (size_t) ato_arg_value_int("connecttimeout");
        arg_forcestmerror = ato_arg_value_bool("forcestmerror");
        arg_securitytokenfile = ato_arg_value("securitytokenfile");
        arg_stmvars = ato_arg_value("stmvars");
        arg_stmrepeat = ato_arg_value_int("stmrepeat");
        arg_stmtemplate = ato_arg_value("stmtemplate");
        arg_stmservice = ato_arg_value("stmservice");
        arg_relyingparty = ato_arg_value("relyingparty");
        arg_stm_rp_p8 = ato_arg_value("stm.rp_p8");

        arg_servicetype = ato_arg_value("servicetype");
        arg_messagetype = ato_arg_value("messagetype");
        arg_agency = ato_arg_value("agency");
        arg_forcecsrerror = ato_arg_value_bool("forcecsrerror");
        arg_mtomthreshold = (size_t) ato_arg_value_int("mtomthreshold");
        arg_runtest = ato_arg_value("runtest");
        arg_busdoclist = ato_arg_value("busdoclist");
        arg_vurilist = ato_arg_value("vurilist");
        arg_govgenid = ato_arg_value("govgenid");
        arg_busgenid = ato_arg_value("busgenid");
        arg_ssid = ato_arg_value("ssid");
        arg_csrskip = ato_arg_value_bool("csrskip");
        arg_csrtemplate = ato_arg_value("csrtemplate");
        arg_csrservice = ato_arg_value("csrservice");
}

static bool _streqi(const char *src, const char *src2) {
    if (src == NULL && src2 == NULL) { return TRUE; }
    if ((src == NULL && src2 != NULL) || (src != NULL && src2 == NULL)) { return FALSE; }
    if (strlen(src) != strlen(src2)) { return FALSE; }
        while (*src2 != '\0') {
        if (tolower((int)*src) != tolower((int)*src2)) { return FALSE; }
                src++; src2++;
        }
        return TRUE;
}

static int curloptfunction(void *curl)
{
    CURLcode rc = CURLE_OK;
    if (curl == NULL) { return rc; }
    printf("Checking CURL options: arg_sslversion=[%s] arg_sslcainfo=[%s]\n", arg_sslversion, arg_sslcainfo);
    if (arg_sslversion != NULL) {
        printf("Setting CURLOPT_SSLVERSION to '%s'\n", arg_sslversion);
        if (_streqi(arg_sslversion, "default")) {
            rc = curl_easy_setopt(curl, CURLOPT_SSLVERSION, CURL_SSLVERSION_DEFAULT);
        } else if (_streqi(arg_sslversion, "tlsv1")) {
            rc = curl_easy_setopt(curl, CURLOPT_SSLVERSION, CURL_SSLVERSION_TLSv1);
        } else if (_streqi(arg_sslversion, "sslv2")) {
            rc = curl_easy_setopt(curl, CURLOPT_SSLVERSION, CURL_SSLVERSION_SSLv2);
        } else if (_streqi(arg_sslversion, "sslv3")) {
            rc = curl_easy_setopt(curl, CURLOPT_SSLVERSION, CURL_SSLVERSION_SSLv3);
        } else if (_streqi(arg_sslversion, "tlsv1_0")) {
            rc = curl_easy_setopt(curl, CURLOPT_SSLVERSION, CURL_SSLVERSION_TLSv1_0);
        } else if (_streqi(arg_sslversion, "tlsv1_1")) {
            rc = curl_easy_setopt(curl, CURLOPT_SSLVERSION, CURL_SSLVERSION_TLSv1_1);
        } else if (_streqi(arg_sslversion, "tlsv1_2")) {
            rc = curl_easy_setopt(curl, CURLOPT_SSLVERSION, CURL_SSLVERSION_TLSv1_2);
        } else {
            printf("***Setting CURLOPT_SSLVERSION FAILED (ignoring)***");
        }
    }
    if (arg_sslcainfo != NULL) {
        if (strcmp(arg_sslcainfo, "off") == 0) {
            printf("%s", "Disabling SSL validation");
            rc = curl_easy_setopt(curl, CURLOPT_SSL_VERIFYHOST, 0L);
            rc = curl_easy_setopt(curl, CURLOPT_SSL_VERIFYPEER, 0L);
        } else {
            // careful - we're not catering to platform independence here
            // so the arg_sslcainfo path is OS specific.
            // Don't bother setting if the file doesn't exist
            struct stat sb;
            if (stat(arg_sslcainfo, &sb) == 0) {
                rc = curl_easy_setopt(curl, CURLOPT_CAINFO, arg_sslcainfo);
            }
        }
    }
    return rc;
}

static bool _getcrinfo(int *ecode, ato_Ctx *ctx, ato_Keystore *ks, const char *alias, ato_String **certificate, ato_String **privatekey, const char *pwd)
{
        ato_Credential *cr = NULL;
        int errcode = ATO_ERR_OK;
        bool ok = FALSE;

        cr = ato_ks_credential(ks, alias);
        if (cr == NULL)
                goto finally;

        // Ignore any renewal errors and continue.
    if (ato_cr_isrenewable(ctx, cr)) {
        //if ato_cr_renew(ctx, cr, pwd) != ATO_ERR_OK) {
            //  printf("Warning: renewal for '%s' failed\n", alias);
        //}
    }

        if ((errcode = ato_cr_certificate(ctx, cr, certificate)) != ATO_ERR_OK ||
                        (errcode = ato_cr_privatekey(ctx, cr, privatekey, pwd)) != ATO_ERR_OK)
                goto finally;

        ok = TRUE;
finally:
        *ecode = errcode;
        return ok;
}

/* From "SBR_WIG_WebServicesImplementationGuide_v2.2c-27Aug2010.pdf" section "4.7. ERROR CODING EXAMPLE"
Try
        Send request and get response
        if (MaximumSeverity = Error)
                For each (EventItem: i)
                        Process based on descriptions and locations
        else
                // Success - check for warnings
                If (MaximumSeverity == Warning)
                        For each (EventItem: i)
                                Process based on descriptions and locations
                Else
                        Ensure at least one item with Information severity
                Process SBDM
Catch (exception e)
        If (e.timeout)
                Queue request for resubmission with increasing delay between attempts
                Notify user - no further action required
        else
                Switch (Fault Code)
                        Case Receiver:
                                Switch case subcode
                                Case sbr:Unavailable
                                        Queue request for resubmission with increasing delay between attempts, using - AvailableAfter - information where present
                                        Notify user - no further action required
                                Case sbr:InternalError
                                        Queue request for resubmission with increasing delay between attempts
                                        Notify user
                                        Possible specific action based on subcode, e.g. log error, generate error report
                Case Sender:
                Case default:
                        Report problem to software vendor capturing the subcode and reason
*/

static void _process_error(ato_Ctx *ctx)
{
        ato_Err *err = ato_ctx_err(ctx);
        int errcode = ato_err_code(err);
        if (errcode == ATO_ERR_OK)
                return;
        printf("\n");
        printf("-----------------------------------------\n");

        switch (errcode) {
                case SBR_CSR_ERR_GENERAL:
                        printf("CSR General error: ");
                        break;
                case SBR_CSR_ERR_NETSENDER:
                        printf("CSR Sender error: ");
                        break;
                case SBR_CSR_ERR_NETRECEIVER:
                        printf("CSR Receiver error: ");
                        break;
                case SBR_CSR_ERR_NETUNAVAILABLE:
                        printf("CSR Remote Service not available: ");
                        break;
                case SBR_CSR_ERR_NETCOMMS:
                        printf("CSR Network error: ");
                        break;
                case SBR_CSR_ERR_NETTIMEOUT:
                        printf("CSR Remote Service timeout: ");
                        break;
        }

        for (; err; err = ato_err_inner(err)) {
                errcode = ato_err_code(err);
                printf("%s\n", ato_err_msg(err));
                if (errcode == ATO_ERR_NET_SOAP) {
                        ato_ErrSoap *soaperr = ato_err_custom(err);
                        if (soaperr && ato_errsoap_issoaperr(soaperr)) {
                                ato_ErrSoapCode *code = NULL;
                                printf("  node=[%s]\n", ato_errsoap_node(soaperr));
                                printf("  reason=[%s]\n", ato_errsoap_reason(soaperr));
                                printf("  detail=[%s]\n", ato_errsoap_detail(soaperr));
                                for (code = ato_errsoap_code_firstv(soaperr); code; code = ato_errsoap_code_nextv(soaperr)) {
                                        printf("  code value=[%s] ns=[%s]\n", ato_errsoap_code_value(code), ato_errsoap_code_ns(code));
                                }
                        } else {
                                printf("  SOAP error details not available\n");
                        }
                }
                printf("-----------------------------------------\n");
        }
        printf("\n");
}

static void _process_eventitems(sbr_Response *response)
{
        size_t indent = 1;
        sbr_Sbdm *sbdm = NULL;
        size_t i = 1, icount = 0;

        icount = sbr_res_sbdm_count(response);
        for (sbdm = sbr_res_sbdm_firstv(response); sbdm; sbdm = sbr_res_sbdm_nextv(response), i++) {
                sbr_MsgEventItem *ei = NULL;
                size_t j = 1, jcount = sbr_sbdm_eventitem_count(sbdm);
                printf("SBDM %" PRINTF_SIZET "u of %" PRINTF_SIZET "u\n", i, icount);
                for (ei = sbr_sbdm_eventitem_firstv(sbdm); ei; ei = sbr_sbdm_eventitem_nextv(sbdm), j++) {
                        dump_eventitem(stdout, &indent, ei, j, jcount, FALSE);
                }
        }
}

static void _process_csrresponse(ato_Ctx *ctx, sbr_Response *response)
{
        if (sbr_res_ismaxseveritycode(response, "Error")) {
                printf("maxseveritycode of %s found\n", "Error");
                _process_eventitems(response);
        } else {
                if (sbr_res_ismaxseveritycode(response, "Warning")) {
                        printf("maxseveritycode of %s found\n", "Warning");
                } else if (sbr_res_ismaxseveritycode(response, "Information")) {
                        printf("maxseveritycode of %s found\n", "Information");
                } else {
                        printf("maxseveritycode of %s found\n", "?unknown?");
                }
                _process_eventitems(response);
                dump_response(ctx, response, "sbrcsr_response_dump.txt");
        }
}

static int _submitrequest(ato_StmTkn *st, sbr_Request *request, sbr_Response **response, ato_String *certificate, ato_String *privatekey)
{
        int errcode = ATO_ERR_OK;
        {
        ato_String *pt = ato_stmtkn_prooftoken(st);
        ato_String *a = ato_stmtkn_assertion(st);
        size_t mtomthresh = 0;

        printf("sbr_req_mtom_size is %" PRINTF_SIZET "u.\n", sbr_req_mtom_size(request));
        sbr_req_mtom_setsize(request, arg_mtomthreshold);
        mtomthresh = sbr_req_mtom_size(request);
        if(mtomthresh != arg_mtomthreshold){
                printf("mtom threshhold didn't stick.\n");
                goto finally;
        }
        printf("NEW sbr_req_mtom_size is %" PRINTF_SIZET "u.\n", mtomthresh);

    if (arg_ssid != NULL) {
        printf("Setting softwareSubscriptionId to %s.\n", arg_ssid);
        sbr_req_ssid_set(request, arg_ssid);
    }

        if ((errcode = sbr_req_submit(g_ctx, request, response, certificate, privatekey, pt, a)) != ATO_ERR_OK)
                goto finally;
        }
        {
                FILE *fp = NULL;
                char *buffer = NULL;
                if ((errcode = sbr_res_xml(g_ctx, *response, &buffer)) != ATO_ERR_OK)
                        goto finally;
                fp = fopen("console-response.xml", "wb");
                fwrite(buffer, sizeof(char), strlen(buffer), fp);
                fclose(fp);
                ato_free(buffer);
        }
finally:
        return errcode;
}

/* removal not currently implemented
static int _amend_req_attachments(ato_Ctx *ctx, sbr_Doc *doc)
{
        int errcode = ATO_ERR_OK;
        if (doc) {
                ato_String *b64 = NULL;
                sbr_DocA *attachment = NULL;
                const char *a_content = "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";
                ato_str_createconst(&b64, a_content, strlen(a_content), TRUE);
                errcode = sbr_doc_attachment_add(ctx, doc, &attachment, "SMP-PDF.pdf", "SMP-PDF.pdf", "application/pdf");
                if (errcode != ATO_ERR_OK) return errcode;
                errcode = sbr_doca_setcontent(ctx, attachment, b64, ATO_PARAMACTION_COPY);
                if (errcode != ATO_ERR_OK) return errcode;
                errcode = sbr_doc_attachment_add(ctx, doc, &attachment, "SMP2-PDF.pdf", "SMP2-PDF.pdf", "application/pdf");
                if (errcode != ATO_ERR_OK) return errcode;
                errcode = sbr_doca_setcontent(ctx, attachment, b64, ATO_PARAMACTION_COPY);
                ato_str_free(b64);
        }
        return errcode;
}
static bool _test_add_remove_attachment(sbr_Request *request)
{
        bool res= FALSE;
        int errcode     = ATO_ERR_OK;
        sbr_Sbdm *sbdm = NULL;
        sbr_Doc *doc = NULL;
        sbr_DocA *att = NULL;
        int num0 = 0;
        int num1 = 0;
        int num2 = 0;

        sbdm = sbr_req_sbdm_firstv(request);
        assert(sbdm != NULL);
        doc = sbr_sbdm_doc_firstv(sbdm);
        num0 = sbr_doc_attachment_count(doc);
        printf("_test_add_remove_attachment: before adding an attachment, number of attachments is %d.\n", num1);
        if(num0 < 0)
                goto finally;
        errcode = _amend_req_attachments(g_ctx, doc);
        if (errcode != ATO_ERR_OK)
                goto finally;
        num1 = sbr_doc_attachment_count(doc);
        printf("_test_add_remove_attachment: after adding an attachment, number of attachments is %d.\n", num1);
        if(num1 < num0)
                goto finally;
        dump_request(g_ctx, request, "req_attachment_added.txt");
        att = sbr_doc_attachment_firstv (doc);
        if(att == NULL)
                goto finally;
        errcode = sbr_doc_attachment_remove (g_ctx, doc, att);
        if (errcode != ATO_ERR_OK)
                goto finally;
        num2 = sbr_doc_attachment_count(doc);
        printf("_test_add_remove_attachment: after removing first attachment, number of attachments is %d.\n",num2);
        dump_request(g_ctx, request, "req_attachment_removed.txt");
        if(num2 != num1-1)
                goto finally;

        //Remove another attachment
        att = sbr_doc_attachment_firstv (doc);
        if(att == NULL)
                goto finally;
        errcode = sbr_doc_attachment_remove (g_ctx, doc, att);
        if (errcode != ATO_ERR_OK)
                goto finally;
        num2 = sbr_doc_attachment_count(doc);
        printf("_test_add_remove_attachment: after removing first attachment, number of attachments is %d.\n",num2);
        dump_request(g_ctx, request, "req_next_attachment_removed.txt");
        if(num2 != num0)
                goto finally;

        res=TRUE;
        finally:

        return res;
}
static bool _test_add_remove_doc(sbr_Request *request)
{
        bool res= FALSE;
        int errcode     = ATO_ERR_OK;
        int numdocs1 = 0;
        int numdocs2 = 0;
        int numdocs3 = 0;
        //ato_Iterator *iter = NULL;
        sbr_Sbdm *sbdm = NULL;
        sbr_Doc *doc = NULL;
        sbr_Doc *tmpdoc = NULL;
        time_t t;
        const char *lcontent = "\
<xbrli:xbrl xmlns:xbrli=\"http://www.xbrl.org/2003/instance\" xmlns:link=\"http://www.xbrl.org/2003/linkbase\" xmlns:xlink=\"http://www.w3.org/1999/xlink\"> \
<link:schemaRef xlink:type=\"simple\" xlink:href=\"http://sbr.gov.au/taxonomy/sbr_au_reports/\"/> \
Simple xbrl instance to use for ping testing. Anything can go in here for a ping... \
</xbrli:xbrl>";
        const char *validationuri = "http://sbr.gov.au/taxonomy/sbr_au_reports/";
        const char *governmentid = NULL;
        const char *businessid =  "66c6109b-f35e-44d1-8178-4a1b358a77e5";

        sbdm = sbr_req_sbdm_firstv(request);
        assert(sbdm != NULL);

        numdocs1 = sbr_sbdm_doc_count(sbdm);
        printf("_test_add_remove_doc initial number of docs is %d\n", numdocs1);

        errcode = sbr_sbdm_doc_add(g_ctx, sbdm, &doc, validationuri, governmentid, businessid, time(&t));
        if (errcode != ATO_ERR_OK)
                goto finally;
        errcode = sbr_doc_setcontent(g_ctx, doc, lcontent, ATO_PARAMACTION_COPY);
        if (errcode != ATO_ERR_OK)
                goto finally;
        numdocs2 = sbr_sbdm_doc_count(sbdm);
        printf("_test_add_remove_doc number of docs after add is %d\n", numdocs2);

        if(numdocs1 != numdocs2-1)
                goto finally;
        dump_request(g_ctx, request, "request_with_added_doc.txt");

                tmpdoc = sbr_sbdm_doc_firstv(sbdm);
        while(tmpdoc != NULL){
                printf("_test_add_remove_doc doc %d createtime is %s.\n", sbr_doc_sequencenr(tmpdoc), sbr_doc_createtime(tmpdoc));
                tmpdoc = sbr_sbdm_doc_nextv(sbdm);
        }

        printf("_test_add_remove_doc doc: about to remove a document.\n");
        errcode = sbr_sbdm_doc_remove(g_ctx, sbdm, doc);
        printf("_test_add_remove_doc doc: done removed first document.\n");
        if (errcode != ATO_ERR_OK)
                goto finally;
        numdocs3 = sbr_sbdm_doc_count(sbdm);
        if(numdocs1 != numdocs3)
                goto finally;
        dump_request(g_ctx, request, "request_with_doc_removed.txt");

        res = TRUE;
        finally:

        return res;
}*/

static bool _test_more_api(sbr_Request *request, sbr_Csr *csr)
{
        bool res = FALSE;
        size_t nettimeout = 0;
        size_t connecttimeout = 0;
        char *buf =NULL;
        int errcode     = ATO_ERR_OK;

        printf("ato_stm_default_templatefile is:%s\n", sbr_csr_default_templatefile());

        errcode = sbr_req_xml (g_ctx, request, &buf);
        if(errcode != ATO_ERR_OK){
                printf("sbr_req_xml call errored out %d.\n", errcode);
                goto finally;
        }
        printf("Here is the current request xml:\n");
        printf("%s\n", buf);
        printf("\n");

        nettimeout = sbr_csr_timeout(g_ctx, csr);
        connecttimeout = sbr_csr_timeoutconnect(g_ctx, csr);
        printf("Total network time out value is %" PRINTF_SIZET "u seconds, connect time out is %" PRINTF_SIZET "u.\n", nettimeout, connecttimeout);
        sbr_csr_settimeout(g_ctx, csr, arg_nettimeout);
        nettimeout = sbr_csr_timeout(g_ctx, csr);
        sbr_csr_settimeoutconnect (g_ctx, csr, arg_connecttimeout);
        connecttimeout = sbr_csr_timeoutconnect(g_ctx, csr);
        if(nettimeout != arg_nettimeout || connecttimeout != arg_connecttimeout){
                printf("time outs failed to stick!\n");
                goto finally;
        }
        printf("Total network time out value is NOW %" PRINTF_SIZET "u seconds and connect time out %" PRINTF_SIZET "u.\n", nettimeout, connecttimeout);


        res = TRUE;
        finally:
        if (buf)
                ato_free(buf);
        return res;
}

//Called if runtest=2 is supplied.
static bool _extra_tests_2(sbr_Request *request, sbr_Csr *csr)
{
        bool res = TRUE;
        //res = _test_add_remove_doc(request);
        //if(res == TRUE) res = _test_add_remove_attachment(request);
        if(res == TRUE) res = _test_more_api(request, csr);
        return res;
}

static bool _run_test01(void)
{
        bool ok = FALSE;
        int errcode = ATO_ERR_OK;
        sbr_Csr *csr = NULL;
        sbr_Request *request = NULL;
        sbr_Response *response = NULL;
        ato_StmTkn *st = NULL;
        ato_Stm *stm = NULL;
        ato_Keystore *ks = NULL;
        ato_String *certificate = NULL;
        ato_String *privatekey = NULL;
        const char *ksdir = NULL, *kspwd = NULL, *ksalias = NULL, *messagetype = NULL, *agency = NULL;

        printf("\n\n");

        ksdir = (arg_ksdir != NULL ? arg_ksdir : _DEFAULT_ksdir);
        kspwd = (arg_kspwd != NULL ? arg_kspwd : _DEFAULT_kspwd);
        ksalias = (arg_ksalias != NULL ? arg_ksalias : _DEFAULT_ksalias);
        messagetype = (arg_messagetype != NULL ? arg_messagetype : _DEFAULT_messagetype);
        agency = (arg_agency != NULL ? arg_agency : _DEFAULT_agency);

       if (arg_servicetype == NULL)
                errcode = sys_load(g_ctx, &ks, ksdir, arg_ksfile, &stm, arg_stmservice, arg_stmtemplate, &csr, arg_csrservice, arg_csrtemplate, "lodge");
        else
                errcode = sys_load(g_ctx, &ks, ksdir, arg_ksfile, &stm, arg_stmservice, arg_stmtemplate, &csr, arg_csrservice, arg_csrtemplate, arg_servicetype);
        if (errcode != ATO_ERR_OK)
                goto finally;

        if (!_getcrinfo(&errcode, g_ctx, ks, ksalias, &certificate, &privatekey, kspwd)) {
                printf("Failed to find alias=%s\n", ksalias);
                goto finally;
        }

        if (arg_securitytokenfile != NULL && *arg_securitytokenfile != '\0') {
                char *buf = NULL;
                ato_String *atorstr = NULL;
                size_t len = utl_loadfile(&buf, arg_securitytokenfile, FALSE);
                assert(len > 0);
                ato_str_create(&atorstr, buf, len, FALSE);
                errcode = ato_stm_loadst(g_ctx, stm, &st, atorstr);
                ato_str_free(atorstr);
                if (errcode != ATO_ERR_OK)
                        goto finally;
                ato_log_msg(g_log, ATO_LOG_INFO, arg_loglevel, "", "", "", "Serialised ATO RSTR loaded from file %s", arg_securitytokenfile);
        } else {
                const char *ato_rstr_filename = "AtoSecurityToken.xml";
                ato_String *pt = NULL;
                FILE *fp = NULL;

        printf("Getting sts token.\n");
                if ((errcode = ato_stm_stsissue(g_ctx, stm, &st, certificate, privatekey, arg_relyingparty)) != ATO_ERR_OK)
                        goto finally;
                //This will reuse cached value - based on hash(template+certificate+relyingpartyurl)
                //if ((errcode = ato_stm_stsissue(g_ctx, stm, &st, certificate, privatekey, arg_relyingparty)) != ATO_ERR_OK)
                       // goto finally;

                pt = ato_stmtkn_xml(st);
                fp = fopen(ato_rstr_filename, "w");
                // Write "SecurityToken" to file - this shows it is "(de)seriablizable".
                if (fp != NULL) {
                        fwrite(ato_str_value(pt), sizeof(char), ato_str_len(pt), fp);
                        fclose(fp);
                }
                ato_log_msg(g_log, ATO_LOG_INFO, arg_loglevel, "", "", "", "Serialised ATO RSTR written to file %s", ato_rstr_filename);
        }

        ato_log_msg(g_log, ATO_LOG_INFO, arg_loglevel, "", "", "", "stkey=%s", ato_stmtkn_key(st));
        ato_log_msg(g_log, ATO_LOG_INFO, arg_loglevel, "", "", "", "SAMLID=%s", ato_stmtkn_samlid(st));

        if (arg_csrskip) {
            printf("Skipping CSR.\n");
            ok = TRUE;
            goto finally;
        }

        if ((errcode = sbr_req_create(g_ctx, &request, csr)) != ATO_ERR_OK)
                goto finally;

        printf("\n\n");

        if ((errcode = amend_request(g_ctx, request, messagetype, agency, arg_busdoclist, arg_vurilist, arg_govgenid, arg_busgenid, arg_forcecsrerror)) != ATO_ERR_OK)
                goto finally;

        if ((arg_runtest != NULL) && strcmp(arg_runtest,"2")==0){
                // Do the extra interface tests before submitting.
                if (!_extra_tests_2(request, csr))
                        goto finally;
        }
        printf("Submitting CSR document.\n");
        errcode = _submitrequest(st, request, &response, certificate, privatekey);
        dump_request(g_ctx, request, "sbrcsr_request_dump.txt");
        if (errcode != ATO_ERR_OK)
                goto finally;
        _process_csrresponse(g_ctx, response);
        printf("\n\n");

        //errcode = amend_response(g_ctx, response); // to test that amending responses fails correctly
        //printf("\n\n");
        if (errcode != ATO_ERR_OK)
                goto finally;

        ok = TRUE;
finally:
        ato_str_free(certificate);
        ato_str_free(privatekey);
        sbr_req_free(request);
        sys_free(g_ctx, ks, stm, csr);
        if (errcode > ATO_ERR_OK) {
                _process_error(g_ctx);
                ato_errh_handler(g_ctx, errcode);
        }
        return ok;
}

static bool _init_tests(void)
{
        int errcode = ATO_ERR_OK;

        if (arg_organisation == NULL || arg_organisation[0] == '\0') { errcode = -1; goto finally; }
        ato_si_set(arg_organisation, _EXE_DESCRIPTION, _EXE_VERSION, _EXE_TIMESTAMP, _EXE_SRC, _EXE_HASH);
        ato_ctx_create(&g_ctx, NULL);
        errcode = init(g_ctx);

        if (errcode != ATO_ERR_OK) {
                ato_Err *err = ato_ctx_err(g_ctx);
                printf("Error: %d (%s) %s: %s\n", ato_err_code(err), ato_err_codename(err), ato_err_codemsg(err), ato_err_msg(err));
                goto finally;
        }

        ato_log_setlogfn(customlogger); //ato_log_setlogfn(verysimplelogger);

        g_log = ato_ctx_log_set(g_ctx, "", "2", NULL, "w");

        // Set log level for all modules
        ato_setloglevel(NULL, 0, arg_loglevel);
        // Override CR as it can generate too much if there are many credentials
        ato_setloglevel(ATO_AKM_LIBRARY, ATO_AKM_MODULEID_CR, ATO_LOG_WARN);
        // Some modules dump extra files useful for debugging
        ato_setlogfiledump(NULL, 0, arg_logfd);
        // Extra log settings for application network traffic
        if (arg_netlogflag)
                ato_bnet_setdebugflag(ato_bnet_names2flag(arg_netlogflag));

    ato_ctx_cachec_set(g_ctx, "stm.rp_p8", arg_stm_rp_p8);

finally:
        return errcode == ATO_ERR_OK;
}
static void _deinit_tests(void)
{
        deinit(g_ctx);
}

static bool _run_tests(void)
{
        bool ok = FALSE;
        bool inited = FALSE;

        if ((inited = _init_tests()) == FALSE)
                goto finally;

        ato_log_start(g_log, ATO_LOG_INFO, arg_loglevel, "", "", "", ato_si_product());

        ato_log_msg(g_log, ATO_LOG_INFO, arg_loglevel, "", "", "", "BASE Version code=%s", ato_base_version());
        ato_log_msg(g_log, ATO_LOG_INFO, arg_loglevel, "", "", "", "AKM Version code=%s", ato_akm_version());
        ato_log_msg(g_log, ATO_LOG_INFO, arg_loglevel, "", "", "", "STM Version code=%s", ato_stm_version());
        ato_log_msg(g_log, ATO_LOG_INFO, arg_loglevel, "", "", "", "CSR Version code=%s", sbr_csr_version());

        ato_log_msg(g_log, ATO_LOG_INFO, arg_loglevel, "", "", "",
            "SoftwareInfo organisation='%s', product='%s', version='%s', timestamp='%s', source='%s'",
            ato_si_organisation(), ato_si_product(), ato_si_version(), ato_si_timestamp(), ato_si_source());
        printf("%s\n", ato_si_info());

        if (!_run_test01())
                goto finally;

        ok = TRUE;
finally:
        if (inited) ato_log_end(g_log, ATO_LOG_INFO, arg_loglevel, "", "", "", FALSE, ato_si_product());
        _deinit_tests();
        return ok;
}

static void _forceemptyline(void)
{
        printf("\n"); printf("\n"); printf(" \n");
}

static bool _setargs(int argc, const char *argv[])
{
        bool argres = FALSE;
        _addargs();
        if (argc > 1) {
                argres = ato_arg_process(argc, argv);
                if (argres) {
                        _initargs();
                        _forceemptyline();
                        argres = ato_arg_check();
                        _forceemptyline();
                }
                argres = (ato_arg_value_bool("help") == FALSE);
        }
    ato_bnet_setcurloptfn(curloptfunction);
        return argres;
}

int main(int argc, const char *argv[])
{
        bool ok = FALSE;
        setbuf(stdout, NULL);
        if (_setargs(argc, argv)) {
                ato_arg_dump();
                _forceemptyline();
                ok = _run_tests();
                if (!ok) {
                        _forceemptyline();
                        printf("An error occurred - please check logs if any\n");
                }
        } else {
                ato_arg_usage(_EXE_NAME);
        }
        ato_arg_free();
        _forceemptyline();
        return ok == TRUE ? 0: -1;
}
